<?php

namespace Drupal\config_uuid_deterministic\Config;

use Drupal\Core\Config\DatabaseStorage;
use Ramsey\Uuid\Uuid;

class DeterministicActiveStorage extends DatabaseStorage {

  use FieldTableNameChecker;

  private const NAMESPACE = '00000000-0000-0000-0000-000000000000';

  public function write($name, array $data) {
    // Skip UUID remapping for field storage configs that would use hashed table names
    if (!$this->shouldSkipUuidRemapping($name, $data)) {
      $uuid = Uuid::uuid5(self::NAMESPACE, $name)->toString();

      // Clean up stale UUID mappings before writing
      $this->cleanupStaleUuidMappings($name, $uuid);

      // Remove existing uuid if present
      unset($data['uuid']);

      // Reconstruct array with uuid as first element
      $data = ['uuid' => $uuid] + $data;

      $data = ConfigUuidRemapper::remapNested($data, $name);
    }
    return parent::write($name, $data);
  }

  public function read($name) {
    $data = parent::read($name);

    // Return FALSE if no data, following parent behavior
    if ($data === FALSE) {
      return FALSE;
    }

    // Return empty array if null
    $data = $data ?: [];

    // Skip UUID remapping for field storage configs that would use hashed table names
    if (!$this->shouldSkipUuidRemapping($name, $data)) {
      // Always set a deterministic UUID, even if one doesn't exist
      $uuid = Uuid::uuid5(self::NAMESPACE, $name)->toString();

      // Reconstruct array with uuid as first element
      $data = ['uuid' => $uuid] + $data;

      $data = ConfigUuidRemapper::remapNested($data, $name);
    }
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function delete($name) {
    // Clean up UUID mappings when deleting config
    $this->cleanupUuidMappingsOnDelete($name);
    return parent::delete($name);
  }

  /**
   * Clean up stale UUID mappings in the key_value store.
   *
   * This prevents conflicts when recreating config entities with the same name.
   * The issue occurs when a config entity is deleted and recreated - the old UUID
   * mapping remains in the key_value store causing conflicts.
   *
   * @param string $name
   *   The configuration name.
   * @param string $correct_uuid
   *   The correct deterministic UUID for this configuration.
   */
  protected function cleanupStaleUuidMappings($name, $correct_uuid) {
    // Extract entity type from config name (e.g., 'block' from 'block.block.forumsponsor')
    $parts = explode('.', $name);
    if (count($parts) < 2) {
      return;
    }

    $entity_type = $parts[0];

    // Get the key_value store for this entity type
    $key_value_collection = "config.entity.key_store.$entity_type";
    try {
      $storage = \Drupal::keyValue($key_value_collection);
    }
    catch (\Exception $e) {
      // If the collection doesn't exist, nothing to clean up
      return;
    }

    // Get all UUID mappings
    $all_mappings = $storage->getAll();

    foreach ($all_mappings as $key => $value) {
      // Check if this is a UUID key that maps to our config name
      if (strpos($key, 'uuid:') === 0 && is_array($value) && in_array($name, $value)) {
        $mapped_uuid = substr($key, 5);

        // If this isn't the correct UUID, it's stale and should be removed
        if ($mapped_uuid !== $correct_uuid) {
          $storage->delete($key);
        }
      }
    }

    // Ensure the correct UUID is properly mapped
    $storage->set("uuid:$correct_uuid", [$name]);
  }

  /**
   * Clean up UUID mappings when deleting configuration.
   *
   * @param string $name
   *   The configuration name being deleted.
   */
  protected function cleanupUuidMappingsOnDelete($name) {
    // Extract entity type from config name
    $parts = explode('.', $name);
    if (count($parts) < 2) {
      return;
    }

    $entity_type = $parts[0];

    // Get the key_value store for this entity type
    $key_value_collection = "config.entity.key_store.$entity_type";
    try {
      $storage = \Drupal::keyValue($key_value_collection);
    }
    catch (\Exception $e) {
      // If the collection doesn't exist, nothing to clean up
      return;
    }

    // Get all UUID mappings and remove any that reference this config
    $all_mappings = $storage->getAll();

    foreach ($all_mappings as $key => $value) {
      if (strpos($key, 'uuid:') === 0 && is_array($value) && in_array($name, $value)) {
        // Remove this config from the mapping
        $updated_value = array_values(array_diff($value, [$name]));

        if (empty($updated_value)) {
          // If no configs left in the mapping, delete the key entirely
          $storage->delete($key);
        }
        else {
          // Otherwise update with the remaining configs
          $storage->set($key, $updated_value);
        }
      }
    }
  }

}
