<?php

namespace Drupal\config_uuid_deterministic\Config;

/**
 * Checks if a field storage config would use hashed table names.
 *
 * When field names are too long, Drupal generates table names using a hash
 * of the field's UUID. Changing the UUID would break table name resolution.
 */
trait FieldTableNameChecker {

  /**
   * Check if a config name is a field storage that would use hashed tables.
   *
   * @param string $configName
   *   The configuration name (e.g., 'field.storage.node.field_example').
   * @param array $data
   *   The configuration data array.
   *
   * @return bool
   *   TRUE if this field storage would use hashed table names.
   */
  protected function wouldUseHashedTableName(string $configName, array $data): bool {
    // Only check field.storage.* configs
    if (!str_starts_with($configName, 'field.storage.')) {
      return FALSE;
    }

    // Extract entity type and field name from the config
    if (!isset($data['entity_type']) || !isset($data['field_name'])) {
      return FALSE;
    }

    // Use the same constants as Drupal core
    $tableLengthLimit = 48;
    $entityTypeMaxLength = 32;

    $entityType = substr($data['entity_type'], 0, $entityTypeMaxLength);
    $fieldName = $data['field_name'];

    // Check both regular and revision table names
    $regularTableName = $entityType . '__' . $fieldName;
    $revisionTableName = $entityType . '_revision__' . $fieldName;

    // If either table name would exceed the limit, Drupal uses hashing
    return strlen($regularTableName) > $tableLengthLimit ||
      strlen($revisionTableName) > $tableLengthLimit;
  }

  /**
   * Check if UUID remapping should be skipped for this config.
   *
   * @param string $configName
   *   The configuration name.
   * @param array $data
   *   The configuration data.
   *
   * @return bool
   *   TRUE if UUID remapping should be skipped.
   */
  protected function shouldSkipUuidRemapping(string $configName, array $data): bool {
    // Skip UUID remapping for field storage configs that would use hashed table names
    if ($this->wouldUseHashedTableName($configName, $data)) {
      // Log this decision for debugging
      \Drupal::logger('config_uuid_deterministic')->debug(
        'Skipping UUID remapping for @config because it would use hashed table names',
        ['@config' => $configName]
      );
      return TRUE;
    }

    return FALSE;
  }

}