<?php

declare(strict_types=1);

namespace Drupal\config_warning\Form;

use Drupal\Core\Render\Element;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Form\SubformStateInterface;
use Drupal\Core\Executable\ExecutableManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Config Warning settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a new SettingsForm instance.
   *
   * @param \Drupal\Core\Executable\ExecutableManagerInterface $conditionManager
   *   The plugin manager for condition plugins.
   */
  public function __construct(
    private readonly ExecutableManagerInterface $conditionManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('plugin.manager.condition'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'config_warning_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['config_warning.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @throws PluginException
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable warning message'),
      '#default_value' => $this->config('config_warning.settings')->get('enabled'),
      '#description' => $this->t("Should the warning message be shown to the admin user? It's convenient to have this disabled during development and then enable it on production."),
    ];
    $form['warning_message'] = [
      '#type' => 'textarea',
      '#rows' => 3,
      '#title' => $this->t('Warning message'),
      '#default_value' => $this->config('config_warning.settings')->get('warning_message'),
      '#description' => $this->t("Add the warning message you want to display. Maybe you have a policy you can refer people to?"),
    ];
    // @todo Consider dynamically discovering supported condition plugins in the future.
    // For now, only explicitly listed plugins are supported.
    $supported_plugins = ['request_path'];
    $conditions_config = $this->config('config_warning.settings')->get('conditions') ?? [];
    // Loop through supported condition plugins and build their form elements.
    foreach ($supported_plugins as $plugin_id) {
      // If the plugin might be corrupted, skip.
      if (!$this->conditionManager->hasDefinition($plugin_id)) {
        continue;
      }

      // Get the configuration for the supported plugin.
      $plugin_config = $conditions_config[$plugin_id] ?? [];
      $plugin = $this->conditionManager->createInstance($plugin_id, $plugin_config);

      if ($plugin instanceof PluginFormInterface) {
        // Wrap condition settings in a collapsible details element.
        $form['conditions'][$plugin_id] = [
          '#type' => 'details',
          '#title' => $this->t('Page path conditions'),
          '#open' => TRUE,
          '#tree' => TRUE,
        ];

        // Let the plugin populate its own configuration form.
        $form['conditions'][$plugin_id] = $plugin->buildConfigurationForm(
          $form['conditions'][$plugin_id],
          $form_state
        );

        // Overrides labels/descriptions for request_path.
        if ($plugin_id === 'request_path') {
          if (isset($form['conditions'][$plugin_id]['pages'])) {
            $form['conditions'][$plugin_id]['pages']['#title'] = $this->t('Paths');
            $form['conditions'][$plugin_id]['pages']['#description'] = $this->t('Specify pages to show the warning on by using their paths. Enter one path per line. You may use wildcards, such as /admin/structure/block/* to match all.');
          }
          if (isset($form['conditions'][$plugin_id]['negate'])) {
            $form['conditions'][$plugin_id]['negate']['#title'] = $this->t('Exclude matching paths');
            $form['conditions'][$plugin_id]['negate']['#description'] = $this->t('<strong>Note:</strong> If <strong>checked</strong>, the warning will be <strong>hidden</strong> on matching paths. If <strong>unchecked</strong>, the warning will be shown <strong>only</strong> on matching paths.');
          }
        }
      }
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    if (empty($form['conditions']) || !is_array($form['conditions'])) {
      return;
    }

    foreach (Element::children($form['conditions']) as $plugin_id) {
      $plugin = $this->conditionManager->createInstance($plugin_id);

      if ($plugin instanceof PluginFormInterface) {
        $plugin->validateConfigurationForm(
          $form['conditions'][$plugin_id],
          $this->getConditionPluginSubformState($plugin_id, $form, $form_state)
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $conditions_config = [];

    if (!empty($form['conditions']) && is_array($form['conditions'])) {
      foreach (Element::children($form['conditions']) as $plugin_id) {
        $plugin = $this->conditionManager->createInstance($plugin_id);

        if ($plugin instanceof PluginFormInterface) {
          $plugin->submitConfigurationForm(
            $form['conditions'][$plugin_id],
            $this->getConditionPluginSubformState($plugin_id, $form, $form_state)
          );

          $conditions_config[$plugin_id] = $plugin->getConfiguration();
        }
      }
    }

    $this->config('config_warning.settings')
      ->set('warning_message', $form_state->getValue('warning_message'))
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('conditions', $conditions_config)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Returns the subform state for a given condition plugin.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $form
   *   The full form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The full form state.
   *
   * @return \Drupal\Core\Form\SubformStateInterface
   *   The subform state scoped to the plugin.
   */
  private function getConditionPluginSubformState(string $plugin_id, array $form, FormStateInterface $form_state): SubformStateInterface {
    return SubformState::createForSubform($form['conditions'][$plugin_id], $form, $form_state);
  }

}
