<?php

namespace Drupal\config_warning\Hook;

use Drupal\block\BlockListBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\user\Form\UserPermissionsForm;
use Drupal\Core\Executable\ExecutableManagerInterface;

/**
 * Defines form hooks for the config_warning module.
 */
class FormHooks {

  /**
   * Constructs a new FormHooks instance.
   *
   * @param \Drupal\Core\Routing\AdminContext $adminContext
   *   The admin context service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Executable\ExecutableManagerInterface $conditionManager
   *   The plugin manager for condition plugins.
   */
  public function __construct(
    private readonly AdminContext $adminContext,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly MessengerInterface $messenger,
    private readonly ExecutableManagerInterface $conditionManager,
  ) {}

  /**
   * Adds a warning message if the altered form modifies configuration.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   * @param string $form_id
   *   ID of the form to check.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Skip if this is not an admin route.
    if (!$this->adminContext->isAdminRoute()) {
      return;
    }

    $config = $this->configFactory->get('config_warning.settings');

    // Skip warning based on route and negate flag.
    $conditions = $config->get('conditions') ?? NULL;
    if ($this->shouldSkipWarningForRoute($conditions)) {
      return;
    }

    // Skip if warnings are disabled.
    if (!$config->get('enabled')) {
      return;
    }

    $form_object = $form_state->getFormObject();
    $form_alters_config = FALSE;

    // Check for forms implementing getEditableConfigNames().
    if (method_exists($form_object, 'getEditableConfigNames')) {
      $form_alters_config = TRUE;
    }

    // Check if editing a config entity.
    if ($form_object instanceof EntityForm) {
      $entity = $form_object->getEntity();
      if ($entity instanceof ConfigEntityInterface && !$entity->isNew()) {
        $form_alters_config = TRUE;
      }
    }

    // User permissions form modifies configuration.
    if ($form_object instanceof UserPermissionsForm) {
      $form_alters_config = TRUE;
    }

    // Block listing form modifies configuration.
    if ($form_object instanceof BlockListBuilder) {
      $form_alters_config = TRUE;
    }

    // Display the warning message if the form alters configuration.
    if ($form_alters_config) {
      $this->messenger->addWarning($config->get('warning_message'));
    }
  }

  /**
   * Determines if the warning should be skipped based on condition plugins.
   *
   * Each configured condition plugin (e.g., request_path, user_role) is
   * evaluated using its `execute()` method. If **any** condition fails, the
   * warning is skipped.
   *
   * This allows flexible control over where and when warnings appear:
   *
   * - For `request_path`, the `negate` flag is used to determine inclusion
   *   or exclusion logic:
   *     - When `negate` is TRUE (default): matching paths are excluded —
   *       no warning shown.
   *     - When `negate` is FALSE: only matching paths are included —
   *       warning shown only on those paths.
   *
   * - For other condition plugins, their native configuration logic applies
   *   (e.g., match current user roles, languages, etc.).
   *
   * @param array $conditions
   *   An array of condition plugin configuration.
   *
   * @return bool
   *   TRUE if the warning should be skipped for the current route.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  private function shouldSkipWarningForRoute(?array $conditions): bool {
    // If no conditions configured, exit early.
    if (!is_array($conditions) || empty($conditions)) {
      return FALSE;
    }

    foreach ($conditions as $plugin_id => $plugin_config) {
      $condition = $this->conditionManager->createInstance($plugin_id, $plugin_config);
      if (!$condition->execute()) {
        return TRUE;
      }
    }
    return FALSE;
  }

}
