<?php

declare(strict_types=1);

namespace Drupal\Tests\config_warning\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests Settings Form.
 *
 * @group config_warning
 */
final class SettingsFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * {@inheritdoc}
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'config_warning',
    'system',
    'user',
    'path',
    'block',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Creates admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'access administration pages',
      'administer blocks',
    ]);

    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests the basic form submission.
   */
  public function testSettingsFormSubmission(): void {
    // Visits the config warning settings form.
    $this->drupalGet('/admin/config/development/config-warning');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form fields exist.
    // Checkbox for enabling config warning message element name = enabled.
    $this->assertSession()->fieldExists('enabled');
    // Textarea for the warning message element name = warning_message.
    $this->assertSession()->fieldExists('warning_message');
    // Textarea for the path's element name = request_path[pages].
    $this->assertSession()->fieldExists('request_path[pages]');
    // Checkbox for include or exclude paths,
    // element name = request_path[negate].
    $this->assertSession()->fieldExists('request_path[negate]');

    // Fill in the form fields.
    $form = [
      'enabled' => TRUE,
      'warning_message' => 'This is a test warning message.',
      'request_path[pages]' => '/admin/structure/block',
      'request_path[negate]' => TRUE,
    ];

    // Submit form.
    $this->submitForm($form, 'Save configuration');

    // Check for success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Get the current config.
    $config = $this->config('config_warning.settings');

    // Check that the config was saved correctly.
    $this->assertTrue($config->get('enabled'));
    $this->assertEquals('This is a test warning message.', $config->get('warning_message'));
    $this->assertEquals([
      'id' => 'request_path',
      'pages' => '/admin/structure/block',
      'negate' => TRUE,
    ], $config->get('conditions')['request_path']);
  }

  /**
   * Tests disabling warning message.
   *
   * If disabled, message should not show to any
   * configuration form.
   */
  public function testDisableWarning(): void {
    // Visits the config warning settings form.
    $this->drupalGet('/admin/config/development/config-warning');
    $this->assertSession()->statusCodeEquals(200);

    // Fill in the form fields.
    $form = [
      'enabled' => FALSE,
      'warning_message' => 'Should remain saved even if disabled.',
      'request_path[pages]' => '/admin/structure/block',
      'request_path[negate]' => TRUE,
    ];

    // Submit form.
    $this->submitForm($form, 'Save configuration');

    // Check for success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Get the current config.
    $config = $this->config('config_warning.settings');

    // Check if the warning message is disabled.
    $this->assertFalse($config->get('enabled'));

    // Check if the message is match and its saved properly.
    $this->assertEquals('Should remain saved even if disabled.', $config->get('warning_message'));

    // Check if the path we defined has the message.
    $this->drupalGet('/admin/structure/block');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the warning message does NOT appear.
    $this->assertSession()->pageTextNotContains('Should remain saved even if disabled.');
  }

  /**
   * Tests condition plugin.
   *
   * If its toggle-on/off, the message will or will not appear
   * base on the path defined in the settings.
   */
  public function testConditionPlugin(): void {
    // Visits the config warning settings form.
    $this->drupalGet('/admin/config/development/config-warning');
    $this->assertSession()->statusCodeEquals(200);

    // Fill in the form fields.
    $form = [
      'enabled' => TRUE,
      'warning_message' => 'Warning message to show in configuration.',
      'request_path[pages]' => '/admin/structure/block',
      'request_path[negate]' => TRUE,
    ];

    // Submit form.
    $this->submitForm($form, 'Save configuration');

    // Check for success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Visit the /admin/structure/block and check that the warning
    // is NOT visible because the toggle is set to exclude paths.
    $this->drupalGet('/admin/structure/block');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextNotContains('Warning message to show in configuration.');

    // Visit the /admin/structure/block/list/claro
    // the message should show here as we only exclude one path.
    $this->drupalGet('/admin/structure/block/list/claro');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Warning message to show in configuration.');

    // Now we will check if the message will show if
    // the toggle is set to include paths.
    $this->drupalGet('/admin/config/development/config-warning');
    $this->submitForm(['request_path[negate]' => TRUE], 'Save configuration');

    // Check for success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Visit the /admin/structure/block and check that the warning
    // is visible because the toggle is set to include paths.
    $this->drupalGet('/admin/structure/block');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextNotContains('Warning message to show in configuration.');

    // Visit the /admin/structure/block/list/claro
    // the message should NOT show here as we only include one path.
    $this->drupalGet('/admin/structure/block/list/claro');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Warning message to show in configuration.');
  }

  /**
   * Tests form validation.
   *
   * Check if the form validates as expected.
   */
  public function testFormConditionPluginValidation(): void {
    // Visits the config warning settings form.
    $this->drupalGet('/admin/config/development/config-warning');
    $this->assertSession()->statusCodeEquals(200);

    // Fill in the form fields.
    $form = [
      'enabled' => TRUE,
      'warning_message' => 'Testing malformed path.',
      'request_path[pages]' => 'malformed_^path/structure/block',
      'request_path[negate]' => TRUE,
    ];

    // Submit form.
    $this->submitForm($form, 'Save configuration');

    // Check that an error message is shown.
    $this->assertSession()->pageTextContains('The path malformed_^path/structure/block requires a leading forward slash when used with the Pages setting.');

    // Optionally confirm the config was not saved.
    $config = $this->config('config_warning.settings');
    $this->assertNotEquals('malformed_^path/structure/block', $config->get('conditions')['request_path']['pages']);
  }

}
