<?php

declare(strict_types=1);

namespace Drupal\Tests\config_warning\Unit;

use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;
use Drupal\block\BlockListBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\user\Form\UserPermissionsForm;
use Drupal\Core\Executable\ExecutableManagerInterface;
use Drupal\Core\Condition\ConditionInterface;
use Drupal\config_warning\Hook\FormHooks;

/**
 * Tests FormHooks service.
 *
 * @group config_warning
 */
final class FormHooksTest extends UnitTestCase {

  /**
   * Mocked admin context service.
   *
   * Allows calling AdminContext methods and PHPUnit mock methods.
   *
   * @var \Drupal\Core\Routing\AdminContext&MockObject
   */
  private AdminContext&MockObject $adminContext;

  /**
   * Mocked config factory service.
   *
   * Allows calling ConfigFactory methods and PHPUnit mock methods.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface&MockObject
   */
  private ConfigFactoryInterface&MockObject $configFactory;

  /**
   * Mocked messenger service.
   *
   * Allows calling Messenger methods and PHPUnit mock methods.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface&MockObject
   */
  private MessengerInterface&MockObject $messenger;

  /**
   * Mocked plugin manager for condition plugins.
   *
   * Allows calling ExecutableManager methods and PHPUnit mock methods.
   *
   * @var \Drupal\Core\Executable\ExecutableManagerInterface&MockObject
   */
  private ExecutableManagerInterface&MockObject $conditionManager;

  /**
   * The FormHooks service under test.
   *
   * @var \Drupal\config_warning\Hook\FormHooks
   */
  private FormHooks $formHooks;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock the services used by the FormHooks constructor.
    $this->adminContext = $this->createMock(AdminContext::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);

    // Mock condition plugin manager; can be
    // extended for condition plugin tests.
    $this->conditionManager = $this->getMockBuilder(ExecutableManagerInterface::class)
      ->disableOriginalConstructor()
      ->getMock();

    // Instantiate the service with mocked dependencies.
    $this->formHooks = new FormHooks(
      $this->adminContext,
      $this->configFactory,
      $this->messenger,
      $this->conditionManager
    );
  }

  /**
   * Provides test cases for testFormAlter().
   *
   * Each case contains:
   * - input (array):
   *   - isAdmin (bool): Whether the route is an admin route.
   *   - enabled (bool|null): Whether warnings are enabled in config.
   *   - formObject (object|string|null): The form object or placeholder.
   *   - conditionExec (bool, optional): Result of condition plugin execute().
   * - expectedWarn (bool): Whether addWarning() is expected to be called.
   *
   * @return array<int, array{
   *   input: array{
   *     isAdmin: bool,
   *     enabled: bool|null,
   *     formObject: object|string|null,
   *     conditionExec?: bool
   *   },
   *     expectedWarn: bool
   *   }>
   */
  public static function formProvider(): array {
    return [
      // Case 1: non-admin route → warning skipped.
      [
        'input' => [
          'isAdmin' => FALSE,
          'enabled' => NULL,
          'formObject' => NULL,
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => FALSE,
      ],
      // Case 2: admin route but warnings disabled → warning skipped.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => FALSE,
          'formObject' => new class() {},
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => FALSE,
      ],
      // Case 3: admin, enabled, form implements
      // getEditableConfigNames → warning shown.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => TRUE,
          'formObject' => new TestConfigFormStub(),
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => TRUE,
      ],
      // Case 4: admin, enabled, UserPermissionsForm → warning shown.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => TRUE,
          'formObject' => 'user_permissions_form',
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => TRUE,
      ],
      // Case 5: admin, enabled, BlockListBuilder → warning shown.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => TRUE,
          'formObject' => 'block_list_builder',
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => TRUE,
      ],
      // Case 6: admin, enabled, EntityForm with ConfigEntity → warning shown.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => TRUE,
          'formObject' => 'entity_form_with_config_entity',
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => TRUE,
      ],
      // Case 7: admin, enabled, condition plugin
      // returns false → warning skipped.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => TRUE,
          'formObject' => 'user_permissions_form',
          'conditionExec' => FALSE,
        ],
        'expectedWarn' => FALSE,
      ],
      // Case 8: admin, enabled, condition plugin returns true → warning shown.
      [
        'input' => [
          'isAdmin' => TRUE,
          'enabled' => TRUE,
          'formObject' => 'user_permissions_form',
          'conditionExec' => TRUE,
        ],
        'expectedWarn' => TRUE,
      ],
    ];
  }

  /**
   * Test the formAlter() method with multiple scenarios.
   *
   * This test also indirectly covers the logic in
   * shouldSkipWarningForRoute() by varying the 'conditionExec' input.
   * We verify whether addWarning() is called based on admin route,
   * enabled config, form type, and supported plugin.
   *
   * @param array $input
   *   Associative array of input values:
   *     - isAdmin: bool, whether the route is an admin route.
   *     - enabled: bool|null, whether warnings are enabled in config.
   *     - formObject: object|string|null, the form object
   *         being tested (string can be a placeholder).
   *     - conditionExec: bool, optional, result of condition
   *         plugin execute() (default true).
   * @param bool $expectedWarn
   *   Whether addWarning() is expected to be called.
   *
   * @dataProvider formProvider
   *   Data provider for the testFormAlter.
   */
  public function testFormAlter(array $input, bool $expectedWarn): void {
    // Extract input values.
    $isAdmin = $input['isAdmin'];
    $enabled = $input['enabled'];
    $conditionExec = $input['conditionExec'] ?? TRUE;
    $formObject = $input['formObject'];

    // Replace placeholder with a mock for UserPermissionsForm.
    if ($formObject === 'user_permissions_form') {
      $formObject = $this->getMockBuilder(UserPermissionsForm::class)
        ->disableOriginalConstructor()
        ->getMock();
    }

    // Replace placeholder with a mock for BlockListBuilder.
    if ($formObject === 'block_list_builder') {
      $formObject = $this->getMockBuilder(BlockListBuilder::class)
        ->disableOriginalConstructor()
        ->getMock();
    }

    // Replace placeholder with a mock for ConfigEntityInterface.
    if ($formObject === 'entity_form_with_config_entity') {
      $entityMock = $this->createMock(ConfigEntityInterface::class);
      $entityMock->method('isNew')->willReturn(FALSE);

      $formObject = $this->getMockBuilder(EntityForm::class)
        ->disableOriginalConstructor()
        ->getMock();

      // The getEntity() call returns a ConfigEntityInterface that is not new,
      // so $form_alters_config becomes TRUE.
      $formObject->method('getEntity')->willReturn($entityMock);
    }

    // Mock config settings for the form warning service.
    $config = $this->createMock(ConfigEntityInterface::class);
    $config->method('get')->willReturnMap([
      ['enabled', $enabled],
      ['warning_message', 'Test warning message'],
      [
        'conditions',
        [
          'request_path' => [
            'id' => 'request_path',
            'negate' => TRUE,
            'pages' => '',
          ],
        ],
      ],
    ]);

    $this->configFactory
      ->method('get')
      ->with('config_warning.settings')
      ->willReturn($config);

    // Mock admin route check.
    $this->adminContext->method('isAdminRoute')->willReturn($isAdmin);

    // Mock the condition plugin; can control skipping warning.
    $pluginMock = $this->createMock(ConditionInterface::class);
    $pluginMock->method('execute')->willReturn($conditionExec);

    $this->conditionManager
      ->method('createInstance')
      ->willReturn($pluginMock);

    // Mock FormStateInterface to return the tested form object.
    $formState = $this->createMock(FormStateInterface::class);
    $formState->method('getFormObject')->willReturn($formObject);

    // Set expectation for messenger: should add warning or not.
    if ($expectedWarn) {
      $this->messenger->expects($this->once())
        ->method('addWarning')
        ->with('Test warning message');
    }
    else {
      $this->messenger->expects($this->never())
        ->method('addWarning');
    }

    // Call the method under test.
    $form = [];
    $this->formHooks->formAlter($form, $formState, 'test_form');

    // Check that the form array is unchanged (method does not modify it).
    $this->assertSame([], $form);
  }

}

/**
 * Test stub class for config forms used in data providers.
 */
class TestConfigFormStub {

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames(): array {
    return ['test'];
  }

}
