<?php

declare(strict_types=1);

namespace Drupal\configurable_anonymizer\Plugin\FieldAnonymizer;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\configurable_anonymizer\Attribute\FieldAnonymizer;
use Drupal\configurable_anonymizer\FieldAnonymizerPluginBase;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Override;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the field_anonymizer.
 */
#[FieldAnonymizer(
  id: 'uuid',
  label: new TranslatableMarkup('UUID field anonymizer'),
  description: new TranslatableMarkup("Uses Drupal core's UUID service to generate UUIDs as anonymized values."),
)]
class UuidFieldAnonymizer extends FieldAnonymizerPluginBase implements ContainerFactoryPluginInterface {


  public function __construct(protected UuidInterface $uuid, array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  #[Override]
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static($container->get('uuid'), $configuration, $plugin_id, $plugin_definition);
  }

  #[Override]
  public function supportsField(FieldDefinitionInterface $field): bool {
    return $field->getType() === 'string';
  }

  #[Override]
  public function anonymize(FieldItemListInterface $field): void {
    if ($field->isEmpty()) {
      return;
    }
    $length = $field->getFieldDefinition()->getSetting('max_length');
    $values = [];
    for ($delta = 0; $delta < $field->count(); $delta++) {
      $values[$delta] = mb_substr($this->uuid->generate(), 0, $length);
    }
    $field->setValue($values);
  }

}
