<?php

namespace Drupal\configurable_anonymizer\Form;

use Drupal\Component\Utility\NestedArray;
use Drupal\configurable_anonymizer\FieldAnonymizerPluginManager;
use Drupal\Core\Config\Entity\ConfigEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class AnonymizerConfigForm extends ConfigFormBase {

  public function __construct(
    protected EntityFieldManagerInterface $fieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FieldAnonymizerPluginManager $fieldAnonymizerManager,
  ) {
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.field_anonymizer'),
    );
  }

  public function getFormId(): string {
    return 'configurable_anonymizer_config_form';
  }

  protected function getEditableConfigNames(): array {
    return ['configurable_anonymizer.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('configurable_anonymizer.settings');


    $form['#tree'] = TRUE;
    $form['fields'] = [];

    /** @var EntityTypeInterface $entity_type */
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      // Do not attempt to anonymize config entities.
      if ($entity_type instanceof ConfigEntityTypeInterface) {
        continue;
      }
      $form['fields'][$entity_type_id] = [
        '#type' => 'details',
        '#title' => $entity_type->getLabel(),
        '#open' => FALSE,
      ];
      $bundles  = $this->getBundles($entity_type);

      foreach ($bundles as $bundle => $label) {
        $form['fields'][$entity_type_id][$bundle] = [
          '#type' => 'details',
          '#title' => $label,
          '#open' => TRUE,
        ];

        foreach ($this->fieldManager->getFieldDefinitions($entity_type_id, $bundle) as $field) {
          if (in_array($field->getName(), [$entity_type->getKey('id'), $entity_type->getKey('uuid'), $entity_type->getKey('bundle')])) {
            continue;
          }
          $form['fields'][$entity_type_id][$bundle][$field->getName()] = [
            '#type' => 'select',
            '#title' => $this->t('@field_label (@field_name)', ['@field_label' => $field->getLabel(), '@field_name' => $field->getName()]),
            '#default_value' => $config->get("fields.$entity_type_id.$bundle.{$field->getName()}") ?? '',
            '#options' => $this->fieldAnonymizerManager->getPluginOptionsForField($field),
          ];
        }
      }
    }

    return parent::buildForm($form, $form_state);
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Filter out empty values on all levels.
    $values = NestedArray::filter(NestedArray::filter(NestedArray::filter($form_state->getValue('fields'))));
    $this->configFactory->getEditable('configurable_anonymizer.settings')
      ->set('fields', $values)
      ->save();

    parent::submitForm($form, $form_state);
  }

  protected function getBundles(EntityTypeInterface $entity_type): array {
    $bundle_id = $entity_type->getBundleEntityType();
    if (!$bundle_id) {
      return [$entity_type->id() => $entity_type->getLabel()];
    }

    $bundles = [];

    $bundle_entities = $this->entityTypeManager->getStorage($bundle_id)->loadMultiple();

    foreach ($bundle_entities as $bundle) {
      $bundles[$bundle->id()] = $bundle->label();
    }

    return $bundles;
  }

}
