<?php

namespace Drupal\configurable_anonymizer\Service;

use Drupal\configurable_anonymizer\FieldAnonymizerPluginManager;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

class AnonymizerService {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  protected array $fields;
  protected EntityTypeManagerInterface $entityTypeManager;
  protected EntityFieldManagerInterface $fieldManager;
  protected FieldAnonymizerPluginManager $pluginManager;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $field_manager,
    TranslationInterface $string_translation,
    FieldAnonymizerPluginManager $plugin_manager,
  ) {
    $this->fields = $config_factory->get('configurable_anonymizer.settings')->get('fields') ?? [];
    $this->entityTypeManager = $entity_type_manager;
    $this->fieldManager = $field_manager;
    $this->stringTranslation = $string_translation;
    $this->pluginManager = $plugin_manager;
  }

  public function anonymize(): void {
    batch_set($this->anonymizeBatch());
  }

  protected function anonymizeBatch(): array {
    $builder = new BatchBuilder();
    foreach ($this->fields as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle => $fields) {
        $builder->addOperation([$this, 'anonymizeEntitiesOfBundle'], [$entity_type_id, $bundle, $fields]);
      }
    }

    return $builder->toArray();
  }

  public function anonymizeEntitiesOfBundle($entity_type_id, $bundle, $fields, &$context) {
    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    $entity_definition = $this->entityTypeManager->getDefinition($entity_type_id);
    $bundle_key = $entity_definition->getKey('bundle');
    $id_key = $entity_definition->getKey('id');
    $query = $storage->getQuery();
    $query->accessCheck(FALSE);
    if ($bundle_key) {
      $query->condition($bundle_key, $bundle);
    }
    if (!isset($context['sandbox']['total'])) {
      $context['sandbox']['total'] = (clone $query)->count()->execute();
      $context['sandbox']['processed'] = 0;
    }

    $query->sort($id_key);
    $query->range($context['sandbox']['processed'], 25);
    $ids = $query->execute();

    if (!$ids) {
      $context['finished'] = 1;
      return;
    }

    $entities = $storage->loadMultiple($ids);

    foreach ($entities as $entity) {
      $context['sandbox']['processed']++;
      foreach ($fields as $field_name => $plugin_id) {
        if (!$plugin_id || !$entity->hasField($field_name)) {
          continue;
        }

        /** @var \Drupal\Core\Field\FieldItemList $field */
        $field = $entity->get($field_name);
        /** @var \Drupal\configurable_anonymizer\FieldAnonymizerInterface $plugin */
        $plugin = $this->pluginManager->createInstance($plugin_id);
        $plugin->anonymize($field);
      }

      $entity->save();
      $context['finished'] = $context['sandbox']['processed'] / $context['sandbox']['total'];
    }
    $context['message'] = $this->t('Processed @entity_type @current of @total', [
      '@entity_type' => $this->entityTypeManager->getDefinition($entity_type_id)->getLabel(),
      '@current' => $context['sandbox']['processed'],
      '@total' => $context['sandbox']['total'],
    ]);
  }
}
