<?php

declare(strict_types=1);

namespace Drupal\configurable_anonymizer_oidc\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Drupal\oidc\OpenidConnectRealm\OpenidConnectRealmManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Configurable Anonymizer OIDC settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  use RedundantEditableConfigNamesTrait;

  protected OpenidConnectRealmManagerInterface $realmManager;

  #[\Override]
  public static function create(ContainerInterface $container): static {
    $self = parent::create($container);
    $self->setRealmManager($container->get('plugin.manager.openid_connect_realm'));

    return $self;
  }

  public function setRealmManager(OpenidConnectRealmManagerInterface $realmManager): void {
    $this->realmManager = $realmManager;
  }


  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'configurable_anonymizer_oidc_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['configurable_anonymizer_oidc.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['disabled_realms'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Disabled realms'),
      '#description' => $this->t('Fields of users from the selected realms will not be anonymized.'),
      '#config_target' => new ConfigTarget(
        'configurable_anonymizer_oidc.settings',
        'disabled_realms',
        fn ($value) => $value ? array_combine($value, $value) : [],
        fn ($value) => array_keys(array_filter($value))
        ),
      '#options' => $this->getRealmOptions(),
    ];
    return parent::buildForm($form, $form_state);
  }

  protected function getRealmOptions(): array {
    $realms = $this->realmManager->getAll();
    $options = [];
    foreach ($realms as $realm) {
      $options[$realm] = $this->realmManager->getDefinition($realm)['name'];
    }

    return $options;
  }
}
