<?php

namespace Drupal\confirm_unpublish\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\user\UserStorageInterface;
use Drupal\node\NodeStorageInterface;

/**
 * Provides a controller to log unpublish confirmations.
 */
class ConfirmUnpublishController extends ControllerBase {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The alias manager service.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected AliasManagerInterface $aliasManager;

  /**
   * The user storage handler.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected UserStorageInterface $userStorage;

  /**
   * The node storage handler.
   *
   * @var \Drupal\node\NodeStorageInterface
   */
  protected NodeStorageInterface $nodeStorage;

  /**
   * Constructs the ConfirmUnpublishController object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
    AliasManagerInterface $aliasManager,
    UserStorageInterface $userStorage,
    NodeStorageInterface $nodeStorage,
  ) {
    $this->configFactory = $configFactory;
    $this->loggerFactory = $loggerFactory;
    $this->aliasManager = $aliasManager;
    $this->userStorage = $userStorage;
    $this->nodeStorage = $nodeStorage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('path_alias.manager'),
      $container->get('entity_type.manager')->getStorage('user'),
      $container->get('entity_type.manager')->getStorage('node')
    );
  }

  /**
   * Logs the unpublish confirmation event.
   *
   * This logs a message when a user confirms the unpublish action
   * via the confirmation dialog.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request, expected to have 'uid' and 'nid'.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response indicating success.
   */
  public function log(Request $request): JsonResponse {
    $uid = $request->request->get('uid');
    $nid = $request->request->get('nid');

    if ($this->configFactory->get('confirm_unpublish.settings')->get('logging')) {
      $user = $this->userStorage->load($uid);
      $node = $this->nodeStorage->load($nid);

      $username = $user ? $user->getAccountName() : 'Unknown';
      $alias = $node
        ? $this->aliasManager->getAliasByPath('/node/' . $nid)
        : '/node/' . $nid;

      $this->loggerFactory->get('confirm_unpublish')->notice(
        'User %user confirmed unpublish of %alias. This only means they clicked continue on the confirmation box. It does not mean they actually saved the node. Check content log or revision history to verify.',
        [
          '%user' => $username,
          '%alias' => $alias,
        ]
      );
    }

    return new JsonResponse(['status' => 'ok']);
  }

}
