<?php

namespace Drupal\confirm_unpublish\Form;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the settings form for the Confirm Unpublish module.
 */
class ConfirmUnpublishSettingsForm extends ConfigFormBase {

  /**
   * The node type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $nodeTypeStorage;

  /**
   * Constructs the settings form.
   */
  public function __construct($config_factory, EntityStorageInterface $node_type_storage) {
    $this->setConfigFactory($config_factory);
    $this->nodeTypeStorage = $node_type_storage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')->getStorage('node_type')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['confirm_unpublish.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'confirm_unpublish_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('confirm_unpublish.settings');

    $content_types = $this->nodeTypeStorage->loadMultiple();
    $content_type_options = [];
    foreach ($content_types as $type) {
      $content_type_options[$type->id()] = $type->label();
    }

    // Manual handling of text_format (cannot use #config_target).
    $form['alert_text'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Confirmation message'),
      '#description' => $this->t('This message will appear in the unpublish confirmation dialog.'),
      '#default_value' => $config->get('alert_text'),
      '#format' => $config->get('alert_text_format') ?: 'basic_html',
      '#required' => TRUE,
    ];

    // Use #config_target for simple fields.
    $form['allowed_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types to exclude from the confirmation dialog'),
      '#description' => $this->t('Select content types to exclude from the unpublish confirmation dialog. All content types are included by default.'),
      '#options' => $content_type_options,
      '#config_target' => 'confirm_unpublish.settings:allowed_content_types',
    ];

    $form['logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log unpublishing actions to <a href="/admin/reports/dblog" target="_blank">Recent Log Messages</a>.'),
      '#description' => $this->t('This will log the user who confirmed the unpublish action. E.g. "User %user confirmed unpublish of %alias."'),
      '#config_target' => 'confirm_unpublish.settings:logging',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save text_format manually.
    $values = $form_state->getValue('alert_text');
    $this->config('confirm_unpublish.settings')
      ->set('alert_text', $values['value'])
      ->set('alert_text_format', $values['format'])
      ->save();

    // Let ConfigFormBase handle the #config_target fields.
    parent::submitForm($form, $form_state);
  }

}
