<?php

namespace Drupal\conreg_clickup;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;

/**
 * Configure conreg_clickup settings for this site.
 */
class ConregConfigClickUpForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'conreg_config_clickup';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'conreg.clickup',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('conreg.clickup');

    $clientId = $config->get('clickup.client_id');
    $clientSecret = $config->get('clickup.client_secret');

    // Check if code returned from ClickUp...
    $code = $this->getRequest()->query->get('code');
    if (!empty($code)) {
      $token = ConregClickUp::getToken($clientId, $clientSecret, $code);

      $config->set('clickup.token', $token);
      $config->save();

      // Redirect to remove code from URL.
      return $this->redirect('conreg_config_clickup');
    }

    $code = $config->get('clickup.code');
    $token = $config->get('clickup.token');

    $return_url = Url::fromRoute('conreg_config_clickup', [], ['absolute' => TRUE]);
    $url = Url::fromUri(
      'https://app.clickup.com/api',
      [
        'query' => [
          'client_id' => $clientId,
          'redirect_uri' => $return_url->toString(),
        ],
      ]
    );
    $external_link = Link::fromTextAndUrl(t('Authenticate link to ClickUp'), $url);

    $form['conreg_authenticate'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('ClickUp Authorization'),
      '#tree' => TRUE,
    ];

    $form['conreg_authenticate']['info'] = [
      '#type' => 'markup',
      '#prefix' => '<div>',
      '#suffix' => '</div>',
      '#markup' => 'To connect ClickUp, fill out Client ID and Secret below, save, then click this link.',
    ];

    $form['conreg_authenticate']['link'] = [
      '#type' => 'markup',
      '#prefix' => '<div>',
      '#suffix' => '</div>',
      '#markup' => $external_link->toString(),
    ];

    $form['conreg_clickup'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('ClickUp Details'),
      '#tree' => TRUE,
    ];

    $form['conreg_clickup']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $clientId,
    ];

    $form['conreg_clickup']['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret'),
      '#default_value' => $clientSecret,
    ];

    $form['conreg_clickup']['token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access token'),
      '#default_value' => $token,
    ];

    if (!empty($token)) {
      $teams = ConregClickUp::getTeam($token);

      foreach ($teams['teams'] as $team) {
        $form['conreg_clickup_team_' . $team['id']] = [
          '#type' => 'fieldset',
          '#title' => $this->t('Team: @team', ['@team' => $team['name']]),
        ];

        $form['conreg_clickup_lists_' . $team['id']] = [
          '#type' => 'fieldset',
          '#title' => $this->t('Lists for: @team', ['@team' => $team['name']]),
        ];
        $form['conreg_clickup_folders_' . $team['id']] = [
          '#type' => 'fieldset',
          '#title' => $this->t('Folders for: @team', ['@team' => $team['name']]),
        ];
        foreach ($team['members'] as $member) {
          $form['conreg_clickup_team_' . $team['id']]['member_' . $member['user']['id']] = [
            '#markup' => $this->t('@id: @username', [
              '@id' => $member['user']['id'],
              '@username' => $member['user']['username'],
            ]),
            '#prefix' => '<div>',
            '#suffix' => '</div>',
          ];
        }
        $spaces = ConregClickUp::getSpaces($team['id'], $token);
        foreach ($spaces['spaces'] as $space) {
          $lists = ConregClickUp::getLists($space['id'], $token);
          foreach ($lists['lists'] as $list) {
            $form['conreg_clickup_lists_' . $team['id']]['list_' . $list['id']] = [
              '#markup' => $this->t('@id: @listName', ['@id' => $list['id'], '@listName' => $list['name']]),
              '#prefix' => '<div>',
              '#suffix' => '</div>',
            ];
          }
          $folders = ConregClickUp::getFolders($space['id'], $token);
          foreach ($folders['folders'] as $folder) {
            $form['conreg_clickup_folders_' . $team['id']]['folder_' . $folder['id']] = [
              '#markup' => $this->t('Folder: @foldername', ['@foldername' => $folder['name']]),
              '#prefix' => '<h3>',
              '#suffix' => '</h3>',
            ];
            foreach ($folder['lists'] as $folderList) {
              $form['conreg_clickup_folders_' . $team['id']]['list_' . $folderList['id']] = [
                '#markup' => $this->t('@id: @listName', ['@id' => $folderList['id'], '@listName' => $folderList['name']]),
                '#prefix' => '<div>',
                '#suffix' => '</div>',
              ];
            }
          }
        }
      }
    }

    $form['conreg_clickup_test'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('ClickUp Test'),
      '#tree' => TRUE,
    ];

    $form['conreg_clickup_test']['info'] = [
      '#type' => 'markup',
      '#prefix' => '<div>',
      '#suffix' => '</div>',
      '#markup' => 'To create a test task, fill out the details below and press "Create".',
    ];

    $form['conreg_clickup_test']['list_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('List ID'),
    ];

    $form['conreg_clickup_test']['assignees'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Assignee Member IDs'),
    ];

    $form['conreg_clickup_test']['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Task Title'),
    ];

    $form['conreg_clickup_test']['description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Task Description'),
    ];

    $form['conreg_clickup_test']['status'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Task Status'),
    ];

    $form['conreg_clickup_test']['submit_test'] = [
      '#type' => 'submit',
      '#value' => t('Create Task'),
      '#submit' => [[$this, 'createTask']],
      '#attributes' => ['id' => "submitBtn"],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Handler for Create Task button.
   */
  public function createTask(array &$form, FormStateInterface $form_state) {
    $vals = $form_state->getValues();

    $message = '';
    $taskId = ConregClickUp::createTask(
              $vals['conreg_clickup_test']['list_id'],
              $vals['conreg_clickup_test']['assignees'],
              $vals['conreg_clickup_test']['title'],
              $vals['conreg_clickup_test']['description'],
              $vals['conreg_clickup_test']['status'],
              NULL,
              $message);
    if (!empty($taskId)) {
      $this->messenger()->addMessage($this->t('Task has been created with ID of @id', ['@id' => $taskId]));
    }
    else {
      $this->messenger()->addMessage($message);
    }

    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $vals = $form_state->getValues();
    $config = $this->config('conreg.clickup');
    $config->set('clickup.client_id', $vals['conreg_clickup']['client_id']);
    $config->set('clickup.client_secret', $vals['conreg_clickup']['client_secret']);
    $config->set('clickup.token', $vals['conreg_clickup']['token']);
    $config->save();

    parent::submitForm($form, $form_state);
  }

}
