<?php

namespace Drupal\consent_management;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\consent_management\PolicyInterface;
use Drupal\consent_management\PolicyVersionInterface;
use Drupal\consent_management\UserConsentInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\user\UserInterface;
use Drupal\Core\Datetime\DateFormatterInterface;

/**
 * Defines the Data Policy Consent Manager service.
 */
class ConsentManager implements ConsentManagerInterface {

  use StringTranslationTrait;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The policy entity.
   *
   * @var \Drupal\consent_management\PolicyInterface
   */
  protected $entity;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The date formatter
   * 
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new GDPR Consent Manager service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    EntityRepositoryInterface $entity_repository,
    DateFormatterInterface $date_formatter,
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->entityRepository = $entity_repository;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public function needConsent() {
    return $this->isDataPolicy() && !$this->currentUser->hasPermission('bypass consent');
  }

  /**
   * {@inheritdoc}
   */
  public function addCheckbox(array &$form) {
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
  
    $versions = $this->getVersionsFromPolicies(['status' => TRUE]);
    
    $links = [];

    $user_agree_version_ids = $this->getAgreedUserConsents();

    foreach ($versions as $key => $policy) {
      // Get translation for current version if exists.
      if ($policy instanceof EntityInterface) {

        $version_id = $policy->getPolicyVersionId();

        $version_date = $this->getCreatedDateOfPolicyVersion($version_id);

        $policy = $this->entityRepository->getTranslationFromContext($policy);

        // Save the consent formular
        $consent_formula[$key] = $policy->getConsentFormula();
  
        $consent_required[$key] = $policy->getPolicyRequired();

        $policy_label = $policy->label() . ' (' . $version_date . ')';
  
        $links[$key] = Link::createFromRoute($policy_label, 'consent_management.policy', ['id' => $policy->getPolicyVersionId()], [
          'attributes' => [
            'class' => ['use-ajax'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => Json::encode([
              'title' => $policy->label(),
              'width' => 700,
              'height' => 700,
            ]),
            'checked' => in_array($policy->getPolicyVersionId(), $user_agree_version_ids),
          ],
        ]);      
      }
    }

    // Checkboxes always should be under all existing fields.
    $form['account']['consent_management'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#weight' => 110,
    ];

    foreach ($links as $entity_id => $link) {

      $enforce_consent_text = $this->getLinkTextFromToken($consent_formula[$entity_id], $link->toString());

      $form['account']['consent_management']['policy_' . $entity_id] = [
        '#type' => 'checkbox',
        '#title' => $enforce_consent_text,
        '#required' => $consent_required[$entity_id],
        '#weight' => 110 + $entity_id,
        '#default_value' => $link->getUrl()->getOption('attributes')['checked'],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function saveConsent($user_id, $action = NULL, array $values = ['state' => UserConsentInterface::STATE_UNDECIDED]) {
    // This logic determines whether we need to create a new "user_consent"
    // entity or not, depending on whether there are new and active
    // "data_policy" with which the user should agree. Previously, there
    // was a `getState` method for this, but it is not relevant since now we
    // do not have a binding to only one entity.
    // See \Drupal\data_policy\Form\DataPolicyAgreement::submitForm.

    $versions = $this->getVersionsFromPolicies(['status' => TRUE]);

    $versions_user_consent = $this->getVersionsFormUserConsent();

    $user_consents = $this->getUserConsents($user_id);

    $existing_states = $this->getExistingStatesFromUserConsent($user_id);


    switch ($action) {
      case 'visit':
        $state = $this->getStateNumber($values['state']);


        if ($new_versions = $this->newPolicyVersionsDetected()) {
          foreach ($new_versions as $version) {
            if ($version) {
              // Unpublish existing
              $this->unpublishPreviousVersions($version);
              // Create new user consent
              $policy = $this->loadPolicyByVersionId($version);   
              $this->createUserConsent($policy, $user_id, $state);  
            }
          }
        }

        break;
      case 'submit':
        // As there will always be a firts hit at least at "visit",
        // we know on submit it's not our first touch here.
        $first = FALSE;  

        foreach ($values as $value) {
          // Get the current state
          $state = $this->getStateNumber($value['state']);

          // Unpubish pervious versions
          $this->unpublishPreviousVersions($value['entity_id']);

          


          // Only create a new record when we do not have one for that state
          if (
            !$this->isEqualState($state, $value['entity_id'])
            ) {

            // Unpublish previous States
            $this->unpublishPreviousStates($value['entity_id']);

            $policy = $this->loadPolicyByVersionId($value['entity_id']);
            $this->createUserConsent($policy, $user_id, $state);
          }
      


          
        }
        break;
    }
 
  }

  protected function unpublishPreviousStates($policy_version_id) {
    $conditions = [
      'uid' => $this->currentUser->id(),
      'cm_policy_version' => $policy_version_id
    ];

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties($conditions);
    if (!empty($user_consents)) {
      foreach ($user_consents as $consent) {
        $consent->setPublished(FALSE)->save();
      }
    }

  }

  protected function isEqualState($state, $policy_version_id) {
    $conditions = [
      'uid' => $this->currentUser->id(),
      'state' => $state,
      'cm_policy_version' => $policy_version_id
    ];

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties($conditions);
    if (!empty($user_consents)) {
      return TRUE;
    }

    return FALSE;


  }

  protected function isSmallerState($state, $policy_version_id) {

    $storage = $this->entityTypeManager->getStorage('cm_user_consent');
    $existing_versions = $storage
      ->getQuery()
      ->condition('status', 1)
      ->condition('state', $state, '<')
      ->condition('uid', $this->currentUser->id())
      ->condition('cm_policy_version', $policy_version_id)
      ->accessCheck()
      ->execute();

    if (!empty($existing_versions)) {
      return TRUE;
    }

    return FALSE;

  }


  protected function unpublishPreviousVersions($policy_version_id) {
    
    // Get the policy of that version
    $policy = $this->loadPolicyByVersionId($policy_version_id);

    // Load all versions for that policy
    $policy_versions = $this->loadPolicyVersionsByPolicy($policy->id());
    unset($policy_versions[$policy_version_id]);


    if (isset($policy_versions) && !empty($policy_versions)) {

      $storage = $this->entityTypeManager->getStorage('cm_user_consent');


      $existing_versions = $storage
        ->getQuery()
        ->condition('status', 1)
        ->condition('uid', $this->currentUser->id())
        ->condition('cm_policy_version', $policy_versions, 'IN')
        ->accessCheck()
        ->execute();

      if (!empty($existing_versions)) {
        $user_consents = $storage->loadMultiple($existing_versions);
        foreach ($user_consents as $consent) {
          $consent->setPublished(FALSE)->save();
        }
      }
    }
  }

  protected function loadPolicyVersionsByPolicy($policy_id) {

    $versions = [];

    $conditions = [
      'cm_policy' => $policy_id,
      'status' => TRUE,
    ];

    $policy_versions = $this->entityTypeManager->getStorage('cm_policy_version')->loadByProperties($conditions);

    foreach ($policy_versions as $version) {
      $versions[$version->id()] = $version->id();
    }

    return $versions;

  }


  protected function getVersionsFormUserConsent() {

    $versions_user_consent = [];

    $conditions = [
      'status' => TRUE,
      'uid' => $this->currentUser->id(),
    ];

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties($conditions);

    if (!empty($user_consents)) {
      foreach($user_consents as $consent) {
        $versions_user_consent[$consent->getPolicyVersionId()] = $consent->getPolicyVersionId();
      }
    }

    return $versions_user_consent;

  }

  protected function getCurrentVersions() {

    $current_versions = [];

    $conditions = [
      'status' => TRUE,
    ];

    $versions = $this->entityTypeManager->getStorage('cm_policy')->loadByProperties($conditions);

    if (!empty($versions)) {
      foreach($versions as $version) {
        $current_versions[$version->getPolicyVersionId()] = $version->getPolicyVersionId();
      }
    }

    return $current_versions;

  }  

  protected function newPolicyVersionsDetected() {
    $difference = array_diff($this->getCurrentVersions(),$this->getVersionsFormUserConsent());
    return $difference;
  }

  protected function isNewPolicy($policy_version_id) {
    $conditions = [
      'uid' => $this->currentUser->id(),
      'cm_policy_version' => $policy_version_id,
    ];

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties($conditions);


  }


  /**
   * Get the number of bool state.
   *
   * @param bool $state
   *   The state value;.
   *
   * @return int
   *   User consent.
   */
  private function getStateNumber($state) {
    if ($state === TRUE) {
      $state = UserConsentInterface::STATE_AGREE;
    }
    elseif ($state === FALSE) {
      $state = UserConsentInterface::STATE_NOT_AGREE;
    }

    return $state;
  }

  /**
   * {@inheritdoc}
   */
  public function getExistingUserConsents($user_id) {
    return $this->entityTypeManager
      ->getStorage('cm_user_consent')
      ->getQuery()
      ->condition('status', 1)
      ->condition('uid', $user_id)
      ->accessCheck()
      ->execute();
  }

  /**
   * Get meta information for consent user.
   *
   * @param integer $user_id
   *   The user id.
   * @return array
   *   The meta data.
   */
  public function getMetaUserDataByUserId(int $user_id): array {

    $meta_data = [];
    $user = $this->entityTypeManager->getStorage('user')->load($user_id);
    if ($user instanceof UserInterface) {
      $meta_data['meta_user_id'] = $user->id();
      $meta_data['meta_user_uuid'] = $user->uuid();
      $meta_data['meta_user_name'] = $user->getDisplayName();
      $meta_data['meta_user_mail'] = $user->getEmail();
      $meta_data['meta_user_created'] = $user->getCreatedTime();
    }

    return $meta_data;

  }

  /**
   * Create the user_consent entity.
   *
   * @param \Drupal\data_policy\Entity\DataPolicyInterface $data_policy
   *   The data policy entity.
   * @param int $user_id
   *   The user id.
   * @param int $state
   *   The state for consent entity.
   * @param bool $required
   *   Required status.
   */
  private function createUserConsent(PolicyInterface $policy, int $user_id, int $state) {

    $storage = $this->entityTypeManager->getStorage('cm_user_consent')->create();

    // Get user data snapshot
    if ($meta_data = $this->getMetaUserDataByUserId($user_id)) {
      foreach ($meta_data as $field => $value) {
        $storage->set($field, $value);
      }
    }

    $storage->set('cm_policy_version', $policy->getPolicyVersionId());
    $storage->setOwnerId($user_id);
    $storage->set('state', $state);
    $storage->save();

  }

  /**
   * {@inheritdoc}
   */
  public function isDataPolicy(): bool {

    $conditions = [
      'status' => TRUE
    ];

    $data_policies = $this->entityTypeManager->getStorage('cm_policy')->loadByProperties($conditions);

    return !empty($data_policies);

  }

  /**
   * {@inheritdoc}
   */
  public function isRequiredDataPolicies(): bool {

    $conditions = [
      'status' => TRUE,
      'policy_required' => TRUE,
    ];

    $policies = $this->entityTypeManager->getStorage('cm_policy')->loadByProperties($conditions);
 
    foreach ($policies as $key => $policy) {
      if ($policy instanceof PolicyInterface) {
        if (!$this->getAllowedPolicyByPolicyUserRoles($policy->id())) {
          unset($policies[$key]);
        }
      }
    }

    return !empty($policies);

  }

  /**
   * {@inheritdoc}
   */
  public function getActivePolicies(): array {

    $result = [];

    $query = $this->entityTypeManager->getStorage('cm_policy')->getQuery();
    $result = $query
      ->condition('status', TRUE)
      ->accessCheck(TRUE) 
      ->execute();

    return $result;

  }   
  
  /**
   * Get versions form policies
   *
   * @param array $conditions
   *   Array of conditions.
   * @return array
   *   Versions keyed by policy version id.
   */
  public function getVersionsFromPolicies(array $conditions): array {

    $versions = [];

    $cm_policies = $this->entityTypeManager->getStorage('cm_policy')->loadByProperties($conditions);

    if (!empty($cm_policies)) {
      foreach ($cm_policies as $entity) {
        if ($policy_version_id = $entity->getPolicyVersionId()) {
          if ($this->getAllowedPolicyByPolicyUserRoles($policy_version_id, 'policy_version')) {
            $versions[$policy_version_id] = $entity;
          }          
        }
      }
    }

    return $versions;

  }    

  /**
   * {@inheritdoc}
   */  
  public function getAgreedUserConsents(): array {

    $versions = [];

    $query = $this->entityTypeManager->getStorage('cm_user_consent')->getQuery();
    $records = $query
      ->condition('status', TRUE)
      ->condition('state', UserConsentInterface::STATE_AGREE)
      ->condition('uid', $this->currentUser->id())
      ->accessCheck(TRUE) 
      ->execute();

    if (!empty($records)) {
      $cm_user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadMultiple($records);
      foreach ($cm_user_consents as $entity) {
        $versions[$entity->getPolicyVersionId()] = $entity->getPolicyVersionId();
      }
    }

    return $versions;

  }

  /**
   * Get user consents
   *
   * @param int $user_id
   *   The user id.  
   * @param $state
   *   The state or NULL.
   * @return array
   *   The user consent entities.
   */
  public function getUserConsents(int $user_id, $state = NULL): array {

    $user_consents = [];
    $conditions = [];

    $conditions['uid'] = $user_id;
    $conditions['status'] = TRUE;

    if (isset($state)) {
      $conditions['state'] = $state;
    }    

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')
      ->loadByProperties($conditions);

    return $user_consents;

  }

  /**
   * Get the link text form token
   *
   * @param $consent_formula
   *  The consent formula given.
   * @param $link
   *  The link.
   * @return string
   *  The Link text.
   */
  public function getLinkTextFromToken($consent_formula, $link) {
    $link_text = str_replace('[policy:link]', $link, $consent_formula);
    return $link_text;
  }

  /**
   * Get user consent versions
   *
   * @param $user_id
   *   The user id or NULL.
   * @return array
   *   The user consent versions.
   */
  public function getUserConsentsVersions($user_id = NULL): array {

    $user_consent_versions = [];

    if (!isset($user_id)) {
      $user_id = $this->currentUser->id();
    }

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')
      ->loadByProperties([
        'uid' => $user_id,
        'status' => TRUE,
      ]);


    if(!empty($user_consents)) {
      foreach ($user_consents as $entity) {
        if ($entity->getState() !== 2) {
          $user_consent_versions[$entity->id()] = $entity->getPolicyVersionId();
        }    
      }
    }

    return $user_consent_versions;

  }

  /**
   * Check if user has agreed on required policies.
   *
   * @return boolean
   *   TRUE | FALSE.
   */
  public function didUserAgreeOnRequiredPolicies(): bool {

    $user_id = $this->currentUser->id();
    $user_consents = $this->getUserConsents($user_id);

    foreach ($user_consents as $consent) {
      // Find out if the policy is set to required
      $policy_version_id = $consent->getPolicyVersionId();
      $policy = $this->loadPolicyByVersionId($policy_version_id);
      if ($policy instanceof PolicyInterface) {
        if ($this->getAllowedPolicyByPolicyUserRoles($policy->id())) {      
          if ($this->isRequiredPolicy($policy) && $consent->getState() != UserConsentInterface::STATE_AGREE) {
            return FALSE;
          }
        }
      }
    }

    return TRUE;  

  }

  /**
   * Check if user has new policies with no consent.
   *
   * @return boolean
   *   TRUE | FALSE.
   */  
  public function newPoliciesWithNoConsent(): bool {



    $is_new_required = FALSE;

    // Get versions of user consent
    $user_consents_versions = $this->getUserConsentsVersions();

    // Get active policy versions
    $policy_versions = $this->getPolicyVersions();

    $diff = array_diff($policy_versions, $user_consents_versions);
    $is_new_consents = array_keys(array_diff($policy_versions, $user_consents_versions));

    if (!empty($diff) && !empty($is_new_consents)) {
      if ($required = $this->isRequiredPolicyInVersions($is_new_consents)) {
        $is_new_required = $required;
      }
    }

    return $is_new_required;


  }


  /**
   * Check if user has no new policies and
   * no pending required policies.
   *
   * @return boolean
   *   TRUE | FALSE.
   */    
  public function noNewPoliciesAndNoNewPendingRequiredPolicies(): bool {

    // Get current user
    $user_id = $this->currentUser->id();

    // Get versions of user consent
    $user_consents_versions = $this->getUserConsentsVersions($user_id);

    // Get active policy versions
    $policy_versions = $this->getPolicyVersions();

    // Get undecided 
    $user_consent_undecided = $this->getUserConsents($user_id, UserConsentInterface::STATE_UNDECIDED);


    $diff = array_diff($policy_versions, $user_consents_versions);
    $is_new_consents = array_keys(array_diff($policy_versions, $user_consents_versions));

    if (empty($diff) && empty($is_new_consents) && empty($user_consent_undecided)) {

     return TRUE;
    }

    return FALSE;

  }

  /**
   * {@inheritdoc}
   */    
  public function getPolicyVersions(): array {

    $policy_versions = [];

    $policies = $this->entityTypeManager->getStorage('cm_policy')
      ->loadByProperties([
        'status' => TRUE,
      ]);

    if(!empty($policies )) {
      foreach ($policies  as $entity) {
        $policy_version_id = $entity->getPolicyVersionId();
        if ($policy_version_id) {
          if ($this->getAllowedPolicyByPolicyUserRoles($policy_version_id, 'policy_version')) {
            $policy_versions[$policy_version_id] = $policy_version_id;
          }   
        }     
      }
    }

    return $policy_versions;

  }  

  /**
   * Get existing states from user.
   *
   * @param int $user_id
   *   The user id.
   * @return array
   *   The user consent states keyed by user consent id.
   */
  public function getExistingStatesFromUserConsent(int $user_id): array {
    $user_consent_states = [];

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')
      ->loadByProperties([
        'uid' => $user_id,
        'status' => TRUE,
      ]);

    if (!empty($user_consents)) {
      foreach ($user_consents as $entity) {
        $user_consent_states[$entity->id()] = $entity->getState(); 
      }
    }

    return $user_consent_states;

  }

  /**
   * Load policy version by ID.
   *
   * @param int $policy_version_id
   *   The policy version ID.
   * @return object
   *   The policy version entity.
   */
  public function loadPolicyVersionById(int $policy_version_id) {
    return $this->entityTypeManager->getStorage('cm_policy_version')->load($policy_version_id);
  }

  /**
   * Load policy version date.
   *
   * @param int $policy_version_id
   *   The policy version ID.
   * @return string
   *   The policy version date or an empty string.
   */  
  public function getCreatedDateOfPolicyVersion(int $policy_version_id) {
    $policy_version = $this->loadPolicyVersionById($policy_version_id);
    if ($policy_version instanceof PolicyVersionInterface) {
      $version_date = $policy_version->created->value;
      return $this->dateFormatter->format($version_date, 'custom', 'd.m.Y');
    }
    return '';
  }

  /**
   * Check if policy is set to required.
   *
   * @param \Drupal\consent_management\PolicyInterface $policy
   *   The policy entity.
   * @return boolean
   *   TRUE | FALSE.
   */
  public function isRequiredPolicy($policy) {
    return $policy->getPolicyRequired();
  }

  /**
   * Check if policy is required by checking versions.
   *
   * @param $policy_versions
   *   Array of policy versions.
   * @return boolean
   *   TRUE | FALSE.
   */
  public function isRequiredPolicyInVersions($policy_versions) {
    if (isset($policy_versions) && !empty($policy_versions)) {
      foreach($policy_versions as $key => $policy_version_id) {
        $policy = $this->loadPolicyByVersionId($policy_version_id);
        if ($policy instanceof PolicyInterface) {
          return $policy->getPolicyRequired();
        }
      }
    }
  }

  /**
   * Load a policy by version id.
   *
   * @param $version_id
   *   The version ID.
   * @return \Drupal\consent_management\PolicyInterface $policy
   *   The policy entity.
   */
  protected function loadPolicyByVersionId($version_id) {
    $conditions = ['cm_policy_version' => $version_id];
    $policies = $this->entityTypeManager->getStorage('cm_policy')->loadByProperties($conditions);
    return reset($policies);
  }

  /**
   * Load a policy by id.
   *
   * @param $policy_id
   *   The policy ID.
   * @return \Drupal\consent_management\Policy $policy
   *   The policy entity.
   */  
  protected function loadPolicy($policy_id) {
    return $this->entityTypeManager->getStorage('cm_policy')->load($policy_id);
  }

  /**
   * Check if policy is allowed by user role.
   *
   * @param $version_id
   *   The version ID of the policy version.
   * @param $type
   *   The type of the retrieval
   * @return boolean
   *   TRUE | FALSE.
   */
  protected function getAllowedPolicyByPolicyUserRoles($id, $type = NULL): bool {

    $allow = TRUE;

    $policy_user_roles = [];

    // Get user roles of current user
    $current_user_roles = $this->currentUser->getRoles();

    if ($type === 'policy_version') {
      $policy = $this->loadPolicyByVersionId($id);
    }
    else {
      $policy = $this->loadPolicy($id);
    }
    
    if ($policy instanceof PolicyInterface) {
      if ($policy_roles = $policy->getUserRoles()) {
        foreach($policy_roles as $role) {
          $policy_user_roles[] = $role->id();
        }
      }
    } 

    if (!empty($policy_user_roles)) {
      if (array_intersect($current_user_roles,$policy_user_roles)) {        
        $allow = TRUE;
      }
      else {
        $allow = FALSE;
      }
    }

    return $allow;

  }

}

