<?php

declare(strict_types=1);

namespace Drupal\consent_management\Entity;

use Drupal\consent_management\PolicyInterface;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\user\EntityOwnerTrait;
use Drupal\Core\Url;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;

/**
 * Defines the policy entity class.
 *
 * @ContentEntityType(
 *   id = "cm_policy",
 *   label = @Translation("Policy"),
 *   label_collection = @Translation("Policies"),
 *   label_singular = @Translation("policy"),
 *   label_plural = @Translation("policies"),
 *   label_count = @PluralTranslation(
 *     singular = "@count policies",
 *     plural = "@count policies",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\consent_management\PolicyListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\consent_management\PolicyAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\consent_management\Form\PolicyForm",
 *       "edit" = "Drupal\consent_management\Form\PolicyForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *       "revision-delete" = \Drupal\Core\Entity\Form\RevisionDeleteForm::class,
 *       "revision-revert" = \Drupal\Core\Entity\Form\RevisionRevertForm::class,
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *       "revision" = \Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider::class,
 *     },
 *   },
 *   base_table = "cm_policy",
 *   data_table = "cm_policy_field_data",
 *   revision_table = "cm_policy_revision",
 *   revision_data_table = "cm_policy_field_revision",
 *   show_revision_ui = TRUE,
 *   translatable = TRUE,
 *   admin_permission = "administer cm_policy",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "langcode" = "langcode",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "collection" = "/admin/config/people/cm-policy",
 *     "add-form" = "/admin/config/people/cm-policy/add",
 *     "edit-form" = "/admin/config/people/cm-policy/{cm_policy}/edit",
 *     "delete-form" = "/admin/config/people/cm-policy/{cm_policy}/delete",
 *     "delete-multiple-form" = "/admin/config/people/cm-policy/delete-multiple",
 *     "revision" = "/admin/config/people/cm-policy/{cm_policy}/revision/{cm_policy_revision}/view",
 *     "revision-delete-form" = "/admin/config/people/cm-policy/{cm_policy}/revision/{cm_policy_revision}/delete",
 *     "revision-revert-form" = "/admin/config/people/cm-policy/{cm_policy}/revision/{cm_policy_revision}/revert",
 *     "version-history" = "/admin/config/people/cm-policy/cm-policy/{cm_policy}/revisions",
 *   },
 * )
 */
final class Policy extends RevisionableContentEntityBase implements PolicyInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function toUrl($rel = 'canonical', array $options = []) {
    try {
      $url = parent::toUrl($rel, $options);
    }
    catch (UndefinedLinkTemplateException $e) {
      $url = Url::fromUri('route:<nolink>')->setOptions($options);
    }
    return $url;
      
  }  

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

    /**
   * {@inheritdoc}
   */
  public function save() {  

    $storage = parent::save();

    return $storage;

  }  

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDescription(t('Enter the name for your policy.'))
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setRevisionable(TRUE)
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled Policy')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['description'] = BaseFieldDefinition::create('text_long')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Description'))
      ->setDescription(t('Describe your policy in a few words.'))
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'label' => 'above',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner');


    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the policy was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the policy was last edited.'));

    $fields['cm_policy_version'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Policy Version'))
      ->setSetting('target_type', 'cm_policy_version');

    $fields['policy_required'] = BaseFieldDefinition::create('boolean')
      ->setRevisionable(TRUE)
      ->setLabel(t('Policy required'))
      ->setDefaultValue(FALSE)
      ->setSetting('on_label', 'Required Policy')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);
    
    $fields['user_roles'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Limit to user roles'))
      ->setDescription(t('Limit this policy to users of certain roles. Add one or more roles. Multiple roles can be delimited by a comma.'))
      ->setSetting('target_type', 'user_role')
      ->setCardinality(-1)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete_tags',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);


    $fields['consent_formula'] = BaseFieldDefinition::create('text_long')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDefaultValue('I agree to [policy:link]')
      ->setLabel(t('Consent formula'))
      ->setDescription(t('Enter your consent text including a token that opens a modal dialog with the policy. Example: Please agree to our data poliy [policy:link]'))
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'label' => 'above',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE);    

    // For migration needed
    $fields['source_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Source ID'))
      ->setDescription(t('The data policy id.'))
      ->setDisplayOptions('form', array(
        'type' => 'number',
        'settings' => array(
          'display_label' => TRUE,
        ),
      ));

    return $fields;
    
  }

  /**
   * {@inheritdoc}
   */
  public function getPolicyVersionId() {
    return $this->get('cm_policy_version')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setPolicyVersionId($policy_version_id) {
    $this->set('cm_policy_version', $policy_version_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPolicyVersion() {
    if ($this->getPolicyVersionId()) {
      return \Drupal::entityTypeManager()->getStorage('cm_policy_version')->load($this->getPolicyVersionId());
    }
    return NULL;    
  }

  /**
   * {@inheritdoc}
   */
  public function getConsentFormula() {
    return $this->get('consent_formula')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setConsentFormula(string $consent_formula) {
    $this->set('consent_formula', $consent_formula);
    return $this;
  }  


  /**
   * {@inheritdoc}
   */
  public function getPolicyRequired() {
    return $this->get('policy_required')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setPolicyRequired(bool $policy_required) {
    $this->set('policy_required', $policy_required);
    return $this;
  }    


  /**
   * {@inheritdoc}
   */
  public function getUserRoles() {
    return $this->get('user_roles')->referencedEntities();
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus() {
    return (bool) $this->get('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus($status) {
    $this->set('status', $status);
    return $this;
  }  



}
