<?php

declare(strict_types=1);

namespace Drupal\consent_management\Entity;

use Drupal\consent_management\PolicyVersionInterface;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\user\EntityOwnerTrait;
use Drupal\Core\Url;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;

/**
 * Defines the policy version entity class.
 *
 * @ContentEntityType(
 *   id = "cm_policy_version",
 *   label = @Translation("Policy version"),
 *   label_collection = @Translation("Policy versions"),
 *   label_singular = @Translation("policy version"),
 *   label_plural = @Translation("policy versions"),
 *   label_count = @PluralTranslation(
 *     singular = "@count policy versions",
 *     plural = "@count policy versions",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\consent_management\PolicyVersionListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\consent_management\PolicyVersionAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\consent_management\Form\PolicyVersionForm",
 *       "edit" = "Drupal\consent_management\Form\PolicyVersionForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *       "revision-delete" = \Drupal\Core\Entity\Form\RevisionDeleteForm::class,
 *       "revision-revert" = \Drupal\Core\Entity\Form\RevisionRevertForm::class,
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\consent_management\ConsentManagementPolicyVersionsHtmlRouteProvider",
 *       "revision" = \Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider::class,
 *     },
 *   },
 *   base_table = "cm_policy_version",
 *   data_table = "cm_policy_version_field_data",
 *   revision_table = "cm_policy_version_revision",
 *   revision_data_table = "cm_policy_version_field_revision",
 *   show_revision_ui = TRUE,
 *   translatable = TRUE,
 *   admin_permission = "administer cm_policy_version",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "langcode" = "langcode",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "add-form" = "/admin/config/people/cm-policy/{cm_policy}/cm-policy-version/add",
 *     "edit-form" = "/admin/config/people/cm-policy/{cm_policy}/cm-policy-version/{cm_policy_version}/edit",
 *     "delete-form" = "/admin/config/people/cm-policy-version/{cm_policy_version}/delete",
 *     "delete-multiple-form" = "/admin/config/people/cm-policy-version/delete-multiple",
 *     "revision" = "/admin/config/people/cm-policy-version/{cm_policy_version}/revision/{cm_policy_version_revision}/view",
 *     "revision-delete-form" = "/admin/config/people/cm-policy-version/{cm_policy_version}/revision/{cm_policy_version_revision}/delete",
 *     "revision-revert-form" = "/admin/config/people/cm-policy-version/{cm_policy_version}/revision/{cm_policy_version_revision}/revert",
 *     "version-history" = "/admin/config/people/cm-policy-version/{cm_policy_version}/revisions",
 *   },
 * )
 */
final class PolicyVersion extends RevisionableContentEntityBase implements PolicyVersionInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function toUrl($rel = 'canonical', array $options = []) {
    try {
      $url = parent::toUrl($rel, $options);
    }
    catch (UndefinedLinkTemplateException $e) {
      $url = Url::fromUri('route:<nolink>')->setOptions($options);
    }
    return $url;
      
  } 

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);

    $policy = \Drupal::routeMatch()->getParameter('cm_policy');

     $values += [
      'cm_policy' => $policy,
    ];

  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    } 
  }

  /**
   * {@inheritdoc}
   */
  public function save() {  

    $storage = parent::save();

    if ($policy = $this->getPolicy()) {
      // Save the connect policy version
      $policy_version_id = $this->id();
      $policy->setPolicyVersionId($policy_version_id);
      $policy->save();
    }

    return $storage;

  }  

  /**
   * {@inheritdoc}
   */
  public function getPolicyId() {
    return $this->get('cm_policy')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setPolicyId($policy_id) {
    $this->set('cm_policy', $policy_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPolicy() {
    return \Drupal::entityTypeManager()->getStorage('cm_policy')->load($this->getPolicyId());
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus() {
    return (bool) $this->get('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus($status) {
    $this->set('status', $status);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setRevisionable(TRUE)
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled');

    $fields['policy_text'] = BaseFieldDefinition::create('text_long')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Policy'))
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'label' => 'above',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the policy version was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the policy version was last edited.'));

    $fields['cm_policy'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Policy reference'))
      ->setSetting('target_type', 'cm_policy');

    // For migration needed
    $fields['source_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Source ID'))
      ->setDescription(t('The revision id of the data policy.'))
      ->setDisplayOptions('form', array(
        'type' => 'number',
        'settings' => array(
          'display_label' => TRUE,
        ),
      ));

    return $fields;
    
  }

}
