<?php

namespace Drupal\consent_management\Entity;

use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\user\UserInterface;
use Drupal\consent_management\PolicyVersionInterface;
use Drupal\consent_management\UserConsentInterface;

/**
 * Defines the User consent entity.
 *
 * @ingroup consent_management
 *
 * @ContentEntityType(
 *   id = "cm_user_consent",
 *   label = @Translation("User consent"),
 *   label_collection = @Translation("User consents"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "views_data" = "Drupal\consent_management\UserConsentViewsData",
 *     "list_builder" = "Drupal\consent_management\UserConsentListBuilder",
 *     "route_provider" = {
 *       "html" = "Drupal\consent_management\UserConsentHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "cm_user_consent",
 *   admin_permission = "overview user consents",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "uid" = "uid",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *   },
 *   links = {
 *     "collection" = "/admin/reports/cm-user-consents",
 *   }
 * )
 */
class UserConsent extends ContentEntityBase implements UserConsentInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('uid')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPublished() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setPublished($published) {
    $this->set('status', $published);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getState() {
    return $this->get('state')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setState($state) {
    $this->set('state', $state);
    return $this;
  }  

  /**
   * {@inheritdoc}
   */
  public function getPolicyVersionId() {
    return $this->get('cm_policy_version')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getPolicyVersion() {
    return $this->get('cm_policy_version')->entity;
  }  

  /**
   * {@inheritdoc}
   */
  public function setPolicyVersionId($policy_version_id) {
    $this->set('cm_policy_version', $policy_version_id);
    return $this;
  } 

  /**
   * {@inheritdoc}
   */
  public function setPolicyVersion(PolicyVersionInterface $policy_version) {
    $this->set('cm_policy_version', $policy_version->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the User consent entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['cm_policy_version'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Policy Version'))
      ->setSetting('target_type', 'cm_policy_version');

    $fields['state'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('State of consent'))
      ->setReadOnly(TRUE)
      ->setSetting('unsigned', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the User consent is published.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    // We implement useful information about the user in case we loose
    // the user accoumnt due to cancelation. It's a snapshot of the user!

    $fields['meta_user_id'] = BaseFieldDefinition::create('integer')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Meta UID'))
      ->setRequired(TRUE)
      ->setDescription(t('Meta data user id.'));

    $fields['meta_user_uuid'] = BaseFieldDefinition::create('string')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Meta data user UUID'))
      ->setRequired(TRUE)
      ->setDescription(t('Meta data user UUID.'))
      ->setSetting('max_length', 255);

    $fields['meta_user_name'] = BaseFieldDefinition::create('string')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Meta Username'))
      ->setRequired(TRUE)
      ->setDescription(t('Meta data username.'))
      ->setSetting('max_length', 255);

    $fields['meta_user_mail'] = BaseFieldDefinition::create('string')
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setLabel(t('Meta Email'))
      ->setRequired(TRUE)
      ->setDescription(t('Meta data email.'))
      ->setSetting('max_length', 255);   
      
    $fields['meta_user_created'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Meta data user created at'))
      ->setDescription(t('The time that the consent user was created.'));

    $fields['meta_user_cancelled'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Meta data user account canceled at'))
      ->setDescription(t('The time that the consent user was canceled.'));
  
    //$fields['source_id'] = BaseFieldDefinition::create('integer')
      //->setLabel(t('Source ID'))
      //->setDescription(t('ID of the source of migration.'));  

    //$fields['source_id_db'] = BaseFieldDefinition::create('integer')
      //->setLabel(t('Source ID DB'))
      //->setDescription(t('ID of the source of migration db.'));     
      
    $fields['original_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Original ID'))
      ->setDescription(t('ID of the original value for cm_policy_version.'));     
    

    return $fields;

  }

  /**
   * Override of the default label() function to return a human-friendly name.
   *
   * @return \Drupal\Component\Render\MarkupInterface|string
   *   Return user display name.
   */
  public function label() {
    $user = $this->getOwner();
    if ($user && $user->getDisplayName()) {
      return $user->getDisplayName();
    }

    return '';
  }

}
