<?php

namespace Drupal\consent_management\EventSubscriber;

use Symfony\Component\HttpKernel\Event\RequestEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\consent_management\PolicyInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\consent_management\ConsentManagerInterface;

/**
 * Redirection subscriber.
 *
 * @package Drupal\data_policy
 */
class RedirectSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The current active route match object.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The redirect destination helper.
   *
   * @var \Drupal\Core\Routing\RedirectDestinationInterface
   */
  protected $destination;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The Data Policy consent manager.
   *
   * @var \Drupal\consent_management\ConsentManagerInterface
   */
  protected $consentManager;

  /**
   * The module handler interface.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  public $database;

  /**
   * RedirectSubscriber constructor.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current active route match object.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $destination
   *   The redirect destination helper.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\consent_management\ConsentManagerInterface $manager
   *   The Data Policy consent manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler interface.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    RouteMatchInterface $route_match,
    RedirectDestinationInterface $destination,
    AccountProxyInterface $current_user,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    MessengerInterface $messenger,
    ConsentManagerInterface $manager,
    ModuleHandlerInterface $module_handler,
    Connection $database
  ) {
    $this->routeMatch = $route_match;
    $this->destination = $destination;
    $this->currentUser = $current_user;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->consentManager = $manager;
    $this->moduleHandler = $module_handler;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[KernelEvents::REQUEST][] = ['checkForRedirection', '28'];
    return $events;
  }

  /**
   * This method is called when the KernelEvents::REQUEST event is dispatched.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event.
   */
  public function checkForRedirection(RequestEvent $event): void {
  
    // Disable for anon. users
    if (!$this->currentUser->isAuthenticated()) {
      return;
    }

    // Check if a data policy is set.
    if (!$this->consentManager->isDataPolicy()) {
      return;
    }

    // Check if the current route is the data policy agreement page.
    if (($route_name = $this->routeMatch->getRouteName()) === 'consent_management.policy.agreement') {
      // The current route is the data policy agreement page. We don't need
      // a redirect response.
      return;
    }

    $route_names = [
      'entity.user.cancel_form',
      'consent_management.policy',
      'image.style_private',
      'system.403',
      'system.404',
      'system.batch_page.html',
      'system.batch_page.json',
      'system.css_asset',
      'system.js_asset',
      'user.cancel_confirm',
      'user.logout',
      'entity_sanitizer_image_fallback.generator',
    ];

    if (in_array($route_name, $route_names, TRUE)) {
      return;
    }

    if ($this->currentUser->hasPermission('bypass consent')) {
      return;
    }

    //$existing_user_consents = $this->consentManager->getExistingUserConsents($this->currentUser->id());
    $is_required_id = $this->consentManager->isRequiredDataPolicies();
    $user_agreed_on_required = $this->consentManager->didUserAgreeOnRequiredPolicies();

    // Do redirect if the user did not agree on required consents.
    if ($is_required_id && !$user_agreed_on_required) {
      $this->doRedirect($event);
      return;
    }

    // Do redirect if there are new required policies with no consent
    $new_policies_missing_consent = $this->consentManager->newPoliciesWithNoConsent();


    if ($new_policies_missing_consent) {
      $this->doRedirect($event);
      return;
    }

    $no_new_consent_no_pending = $this->consentManager->noNewPoliciesAndNoNewPendingRequiredPolicies();
    if ($no_new_consent_no_pending) {
      return;
    }

    $this->addStatusLink();
  
  }

  /**
   * Do redirect to the agreement page.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event.
   */
  private function doRedirect(RequestEvent $event): void {
    // Set the destination that redirects the user after accepting the
    // data policy agreements.
    $destination = $this->getDestination();

    // Check if there are hooks to invoke that do an override.
    if ($this->moduleHandler->hasImplementations('consent_management_destination_alter')) {
      $implementations = $this->moduleHandler->invokeAll('consent_management_destination_alter', [
        $this->currentUser,
        $this->getDestination(),
      ]);

      $destination = end($implementations);
    }

    $url = Url::fromRoute('consent_management.policy.agreement', [], [
      'query' => $destination->getAsArray(),
    ]);

    $response = new RedirectResponse($url->toString());
    $event->setResponse($response);
  }

  /**
   * Add the status link.
   */
  private function addStatusLink() {
    if ($this->routeMatch->getRouteName() !== 'consent_management.policy.agreement') {
      $link = Link::createFromRoute($this->t('here'), 'consent_management.policy.agreement');
      $this->messenger->addStatus($this->t('We published a new version of the data protection statement. You can review the data protection statement @url.', [
        '@url' => $link->toString(),
      ]));
    }
  }

  /**
   * Get the redirect destination.
   *
   * @return \Drupal\Core\Routing\RedirectDestinationInterface
   *   The redirect destination.
   */
  protected function getDestination(): RedirectDestinationInterface {
    return $this->destination;
  }

}
