<?php

declare(strict_types=1);

namespace Drupal\consent_management\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\consent_management\Migration;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityInterface;

/**
 * Provides a Consent Management form.
 */
final class MigrationForm extends FormBase {

  /**
   * The migration service.
   * 
   * @var \Drupal\consent_management\Migration
   */
  protected $migrationService;
  

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->setMigrationService($container->get('consent_management.migration'));
    return $instance;
  }  

  /**
   * Set the migration service.
   *
   * @param \Drupal\consent_management\Migration $migration
   *   The migration service.
   */
  public function setMigrationService($migration) {
    $this->migrationService = $migration;
  }   

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'consent_management_migration';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['intro'] = [
      '#type' => 'html_tag',
      '#tag' => 'p',
      '#value' => $this->t('Migrate your data policy data into consent management!'),
    ];

    if (\Drupal::moduleHandler()->moduleExists('data_policy')) {

      $form['actions'] = [
        '#type' => 'actions',
        'policy' => [
          '#type' => 'submit',
          '#value' => $this->t('Migrate data policy and revisions'),
        ],
        'user_consent' => [
          '#type' => 'submit',
          '#value' => $this->t('Migrate user consent data'),
        ],
      ];

      $policies = $this->migrationService->policyDataPresent();


      if ($policies) {
        $form['actions']['policy']['#disabled'] = TRUE; 
      } 

      $consents = $this->migrationService->userConsentPresent();

      if ($consents) {
        $form['actions']['user_consent']['#disabled'] = TRUE;  
      }

    }
    else {
      $this->messenger()->addStatus($this->t('You do not have data policy module enabled. No need for any migration!'));  
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // @todo Validate the form here.
    // Example:
    // @code
    //   if (mb_strlen($form_state->getValue('message')) < 10) {
    //     $form_state->setErrorByName(
    //       'message',
    //       $this->t('Message should be at least 10 characters.'),
    //     );
    //   }
    // @endcode
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    //$this->messenger()->addStatus($this->t('The message has been sent.'));
    //$form_state->setRedirect('<front>');

    $enable_batch_policy = FALSE;
    $enable_batch_consent = FALSE; 

    $triggering_element = $form_state->getTriggeringElement();
    $action = $triggering_element['#id'];

    if ($action === 'edit-policy') {

      $data_policies = $this->migrationService->loadAllDataPolicies();

      $operation = [];
  
      if (!empty($data_policies)) {
        foreach ($data_policies as $data_policy) {          
          $operation[] = [
            '\Drupal\consent_management\Batch\DataPolicyMigrationBatch::operation',
            [$data_policy]
          ];
        }
        $enable_batch_policy  = TRUE;
      }

      if ($enable_batch_policy ) {
        $batch = [
          'title' => 'Migrating ...',
          'operations' => $operation,
          'finished' => '\Drupal\consent_management\Batch\DataPolicyMigrationBatch::finished'
        ];
    
        batch_set($batch);
      }
      else {
        \Drupal::messenger()->addMessage(t('No data to process!')); 
      }
  


    }

    if ($action === 'edit-user-consent') {

      $user_consents = $this->migrationService->userConsentPresent();
      if (!$user_consents) {
        $enable_batch_consent = TRUE;
        $operation[] = [
          '\Drupal\consent_management\Batch\UserConsentMigrationBatch::operation', []
        ];
      }       
      
      if ($enable_batch_consent) {  
        $batch = [
          'title' => 'Migrating ...',
          'operations' => $operation,
          'finished' => '\Drupal\consent_management\Batch\UserConsentMigrationBatch::finished'
        ];  
        batch_set($batch);        
      }
      else {
        \Drupal::messenger()->addMessage(t('No data to process!'));         
      }
    }
  }
}
