<?php

namespace Drupal\consent_management\Form;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\consent_management\ConsentManagerInterface;
use Drupal\consent_management\Entity\PolicyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Component\Render\FormattableMarkup;

/**
 * Policy agreement form.
 *
 * @ingroup data_policy
 */
class PolicyAgreement extends FormBase {
  use StringTranslationTrait;

  /**
   * The Data Policy consent manager.
   *
   * @var \Drupal\consent_management\ConsentManagerInterface
   */
  protected $consentManager;

  /**
   * The redirect destination helper.
   *
   * @var \Drupal\Core\Routing\RedirectDestinationInterface
   */
  protected $destination;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * DataPolicyAgreement constructor.
   *
   * @param \Drupal\consent_management\ConsentManagerInterface $manager
   *   The Consent manager.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $destination
   *   The redirect destination helper.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    ConsentManagerInterface $consent_manager,
    RedirectDestinationInterface $destination,
    DateFormatterInterface $date_formatter
  ) {
    $this->consentManager = $consent_manager;
    $this->destination = $destination;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('consent_management.manager'),
      $container->get('redirect.destination'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'consent_management_policy_agreement';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $this->consentManager->addCheckbox($form);
    $this->consentManager->saveConsent($this->currentUser()->id(), 'visit');

    $date = $this->dateFormatter->format(time(), 'html_date');

    $form['date'] = [
      '#theme' => 'status_messages',
      '#message_list' => [
        'info' => [
          [
            '#type' => 'html_tag',
            '#tag' => 'strong',
            '#value' => $this->t('Our data protection statement has been updated'),
          ],
        ],
      ],
      '#weight' => -2,
    ];

    if (!empty($this->config('data_policy.data_policy')->get('enforce_consent'))) {
      $form['data_policy']['#weight'] = 1;

      $link = Link::createFromRoute($this->t('the account cancellation'), 'entity.user.cancel_form', [
        'user' => $this->currentUser()->id(),
      ]);

      $form['not_agree'] = [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('Consent to the data protection statement(s) is required for continuing using this platform. If you do not consent, you will be guided to @url process.', [
          '@url' => $link->toString(),
        ]),
        '#theme_wrappers' => [
          'form_element',
        ],
        '#weight' => -1,
      ];
    }

    $form['actions']['#type'] = 'actions';

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('consent_management');
    $result = [];
    
    if (!empty($values)) {

      foreach ($values as $name => $value) {
        $result[$name] = [
          'value' => $value,
          'required' => $form['account']['consent_management'][$name]['#required'],
          'entity_id' => (int) filter_var($name, FILTER_SANITIZE_NUMBER_INT),
          'state' => !empty($values[$name]),
        ];
      }

      $this->consentManager->saveConsent($this->currentUser()->id(), 'submit', $result);
      
    }

    foreach ($result as $item) {
      // If the user does not agree and it is enforced, we will redirect to
      // the cancel account page.
      if ($item['required'] && !$item['value']) {
        $this->getRequest()->query->remove('destination');

        $form_state->setRedirect('entity.user.cancel_form', [
          'user' => $this->currentUser()->id(),
        ]);
      }
    }

    // If the user agrees or does not agree (but it is not enforced), check if
    // we should redirect to the front page.
    if ($this->destination->get() === '/policy-agreement') {
      $form_state->setRedirect('<front>');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $errors = $form_state->getErrors();

    if (!empty($errors)) {
      $form_state->clearErrors();

      foreach ($errors as $id => $error) {
        /** @var \Drupal\Core\StringTranslation\TranslatableMarkup $g */
        if (strpos($id, 'policy') !== FALSE) {
          $name = Markup::create($error->getArguments()['@name']);
          $rendered_name = new TranslatableMarkup ('@message', array('@message' => $name));
          //\Drupal::logger('consent_management')->notice('NAME: <pre>'.print_r($name, TRUE).'</pre>');
          $form_state->setErrorByName($id, $rendered_name);
          continue;
        }

        $form_state->setErrorByName($id, $error);

      }
    }

    parent::validateForm($form, $form_state);
  }

}
