<?php

declare(strict_types=1);

namespace Drupal\consent_management;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\user\UserInterface;
use Drupal\consent_management\UserConsentInterface;
use Drupal\consent_management\PolicyInterface;
use Drupal\consent_management\PolicyVersionInterface;

/**
 * @todo Add class description.
 */
final class Helper {

  /**
   * Constructs a Helper object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly QueueFactory $queue,
  ) {}

  /**
   * Fetch all platform users
   * 
   * @return array
   *   Array of user ids keyed by user id.
   */
  public function fetchAllPlatformUsers(): array {
    $user_array = [];    
    // Get all users and populate our array
    $users = $this->entityTypeManager->getStorage('user')->loadMultiple();
    foreach ($users as $user) {
      if ($user instanceof UserInterface) {
        $user_array[$user->id()] = $user->id();
      }
    }

    return $user_array;

  }

  /**
   * Check if a policy version exists
   * 
   * @param int policy_version_id
   *   The ID of the policy version.
   * @param int uid
   *   The user ID of the given user.
   * @return bool
   *   TRUE | FALSE.
   */
  public function policyVersionExists(int $policy_version_id, $uid) {

    $conditions = [
      'cm_policy_version' => $policy_version_id,
      'uid' => $uid
    ];

    $policy_versions = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties($conditions);
    if (!empty($policy_versions)) {
      return TRUE;
    }

    return FALSE;

  }

  /**
   * Create policy version for user
   *
   * @param integer $policy_version_id
   *   The policy version id.
   * @param integer $uid
   *   The user id.
   * @return void
   */
  public function createPolicyVersionForUserConsent(int $policy_version_id, int $uid) {

    $data = [
      'cm_policy_version' => $policy_version_id,
      'uid' => $uid,
      'state' => 0,
      'status' => 1
    ];

    try {
      $cm_user_consent = $this->entityTypeManager->getStorage()->create($data);
      $cm_user_consent->save();
    }
    catch(\Exception $exception) {
      \Drupal::logger('consent_management')->warning('Problem creating a user consent: ' . $exception->getMessage());
    }

  }

  public function getActivePolicies() {

    $entities = [];

    $storage = $this->entityTypeManager->getStorage('cm_policy');
    $query =  $storage->getQuery();
    $entity_ids = $query
      ->condition('status', TRUE)
      ->accessCheck(FALSE) 
      ->execute();

    if (!empty($entity_ids)) {
      $entities = $storage->loadMultiple($entity_ids); 
    }

    return $entities;

  }

  /**
   * Update meta data when user has been removed
   *
   * @param integer $user_id
   *   The user id.
   */
  public function updateUserConsentUserMetaData(int $user_id) {
    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties(['uid' => $user_id]);
    if (!empty($user_consents)) {
      foreach ($user_consents as $consent) {
        if ($consent instanceof UserConsentInterface) {
          $consent->set('uid', 1);
          $consent->set('meta_user_cancelled', time());
          $consent->save();
        }
      }
    }
  }

  public function isPolicyVersionActiveInUserConsent(int $policy_version_id) {
    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties(['cm_policy_version' => $policy_version_id]);
    if (!empty($user_consents)) {
      return TRUE;
    }
    return FALSE;
  }

  protected function isMultiplePolicyVersionsActiveInUserConsent(array $policy_versions) {
    $user_consent_data = $this->entityTypeManager
      ->getStorage('cm_user_consent')
      ->getQuery()
      ->condition('cm_policy_version', $policy_versions, 'IN')
      ->accessCheck(FALSE)
      ->execute();

    return !empty($user_consent_data);

  }

  /**
   * Check if policy is active in user consent.
   *
   * @param integer $policy_id
   *   The policy id.
   * @return boolean
   *   TRUE | FALSE.
   */
  public function isPolicyActiveInUserConsent(int $policy_id) {

    $policy_versions = $this->loadPolicyVersionsByPolicyId($policy_id);

    if (!empty($policy_versions)) {
      if ($this->isMultiplePolicyVersionsActiveInUserConsent($policy_versions)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Load a policy by version id.
   *
   * @param $version_id
   *   The version ID.
   * @return \Drupal\consent_management\PolicyInterface $policy
   *   The policy entity.
   */
  protected function loadPolicyByVersionId($version_id) {
    $conditions = ['cm_policy_version' => $version_id];
    $policies = $this->entityTypeManager->getStorage('cm_policy')->loadByProperties($conditions);
    return reset($policies);
  }  

  /**
   * Get all policy versions by policy id.
   *
   * @param $policy_id
   *   The ID of the policy.
   * @return array
   *   The policy version ids.
   */
  protected function loadPolicyVersionsByPolicyId($policy_id): array {

    $policy_version_ids = [];

    $conditions = ['cm_policy' => $policy_id];
    $policy_versions = $this->entityTypeManager->getStorage('cm_policy_version')->loadByProperties($conditions);
    if (!empty($policy_versions)) {
      foreach ($policy_versions as $key => $version) {
        $policy_version_ids[$key] = $key;
      }
    }
    return $policy_version_ids;
  }

  public function getAllEnabledPolicyVersionIdsAsString() {

    $version_ids = [];
    $versions_string = NULL;

    $policies = $this->getActivePolicies();
    if (!empty($policies)) {
      foreach ($policies as $policy) {
        if ($policy instanceof PolicyInterface) {
          $version_ids[] = $policy->getPolicyVersionId();
        }
      }
    }

    if (!empty($version_ids)) {
      $versions_string = implode(',', $version_ids);
    }
  
    return $versions_string;

  }

  
}

