<?php

declare(strict_types=1);

namespace Drupal\consent_management;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Database\Connection;

/**
 * @todo Add class description.
 */
final class Migration {

  /**
   * Constructs a Migration object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly Connection $database,
  ) {}

  /**
   * @todo Add method description.
   */
  public function getAllRelatedRevisions($entity_type_id, $entity) {
    $revisions = [];

    $vids = $this->entityTypeManager->getStorage($entity_type_id)->revisionIds($entity);

    foreach ($vids as $vid) {
      $revisions[$vid] = $this->entityTypeManager->getStorage($entity_type_id)->loadRevision($vid);
    }

    return $revisions;

  }

  /**
   * @todo Add method description.
   */
  public function getCurrentRevision($entity_type_id, $entity) {
    $revisions = [];

    $latest_vid = \Drupal::entityTypeManager()
        ->getStorage($entity_type_id)
        ->getLatestRevisionId($entity->id());

    $current_revision = $this->entityTypeManager->getStorage($entity_type_id)->loadRevision($latest_vid);    
    if ($current_revision instanceof EntityInterface) {
      return $current_revision;
    }

    return NULL;    

  }  

  public function loadAllDataPolicies() {
    $data_policies = $this->entityTypeManager->getStorage('data_policy')->loadMultiple();
    return $data_policies;
  }

  public function policyDataPresent() {
    $policies = $this->entityTypeManager->getStorage('cm_policy')->loadMultiple();
    if (!empty($policies)) {
      return TRUE;
    }
    return FALSE;
  } 
  
  public function userConsentPresent() {
    $query = $this->entityTypeManager->getStorage('cm_user_consent')->getQuery();
    $query->accessCheck(FALSE);
    $user_consents = $query->count()->execute();

    if ($user_consents > 0) {
      return TRUE;
    }
    return FALSE;
  }

  public function loadPolicy(int $entity_id) {
    $policy = $this->entityTypeManager->getStorage('cm_policy')->load($entity_id);
    return $policy;
  }

  public function loadUserConsent($entity_id) {
    $user_consent = $this->entityTypeManager->getStorage('user_consent')->load($entity_id);
    return $user_consent;
  }

  public function loadCmUserConsent($entity_id) {
    $user_consent = $this->entityTypeManager->getStorage('cm_user_consent')->load($entity_id);
    return $user_consent;
  }  

  public function loadCmUserConsentByConditions($conditions) {

    $user_consents = $this->entityTypeManager->getStorage('cm_user_consent')->loadByProperties($conditions);
    if ($user_consent = reset($user_consents)) {
      return $user_consent;
    }

    return NULL;
    
  }  

  public function loadPolicyVersion(int $entity_id) {
    $policy_version = $this->entityTypeManager->getStorage('cm_policy_version')->load($entity_id);
    return $policy_version;
  }  

  public function createPolicy($data) {

    $entity_id = NULL;

    $storage = $this->entityTypeManager->getStorage('cm_policy');
    $entity = $storage->create($data);
    $entity->save();
    $entity_id = $entity->id();
    return $entity_id;

  }

  public function createPolicyVersion($data) {

    $entity_id = NULL;

    $storage = $this->entityTypeManager->getStorage('cm_policy_version');
    $entity = $storage->create($data);
    $entity->save();
    $entity_id = $entity->id();
    return $entity_id;
    
  }  

  public function loadAllUserConsent() {
    $user_consents = $this->entityTypeManager->getStorage('user_consent')->loadMultiple();
    return $user_consents;
  }

  public function getPolicyVersionIdBySourceId($source_id) {
    $policy_versions = $this->entityTypeManager->getStorage('cm_policy_version')->loadByProperties(['source_id' => $source_id]);
    if ($policy_version = reset($policy_versions)) {
      return $policy_version;
    }
    return NULL;
  }

  public function createUserConsent($data) {
    $storage = $this->entityTypeManager->getStorage('cm_user_consent');

    try {
      $entity = $storage->create($data);
      $entity->save();      
    }
    catch (\Exception $exception) {
      $message = $exception->getMessage();
      $debug = [
        'data' => $data,
        'message' => $message
      ];
      \Drupal::logger('consent_management')->warning('<pre>'.print_r($debug, TRUE).'</pre>');
    }

  }

  public function amountOfRecordsMigrateUserConsent() {

    $query = $this->database->select('migrate_user_consent', 'm');
    $query->fields('m', ['id']);
    $number_records = $query->countQuery()
      ->execute()
      ->fetchField();

    return $number_records;

  }

  public function amountOfRecordsUserConsent() {
    $storage = $this->entityTypeManager->getStorage('user_consent');
    $query = $storage->getQuery();
    $query->accessCheck(FALSE);    
    $count = $query->count()->execute();
    return $count;
  }

  public function amountOfRecordsCmUserConsent() {
    $storage = $this->entityTypeManager->getStorage('cm_user_consent');
    $query = $storage->getQuery();
    $query->accessCheck(FALSE);    
    $count = $query->count()->execute();
    return $count;
  }  

  public function amountOfRecordsOldCmUserConsent() {
    
    $storage = $this->entityTypeManager->getStorage('cm_user_consent');
    $query = $storage->getQuery();
    
    $query->condition('created', 1731401034, '<')
          ->accessCheck(FALSE);    

    $count = $query->count()->execute();
    return $count;

  }    

  public function getMigrateUserConsentRange($start, $limit) {

    $query = $this->database->select('migrate_user_consent', 'm');
    $query->fields('m', ['id', 'uuid', 'langcode', 'user_id', 'state', 'status', 'created', 'changed', 'revision_id']);
    $query->range($start, $limit)
          ->orderBy('id');
    
    $result = $query->execute();

    return $result->fetchAllAssoc('id');

  }  

  public function getUserConsentRange($start, $limit) {
    $storage = $this->entityTypeManager->getStorage('user_consent');
    $query = $storage->getQuery();
    $query->accessCheck(FALSE)
          ->sort('id', 'ASC')
          ->range($start, $limit);
    
    $result = $query->execute();

    if (!empty($result)) {
      return $result;
    }

    return NULL;
  }

  public function getCmUserConsentRange($start, $limit) {
    $storage = $this->entityTypeManager->getStorage('cm_user_consent');
    $query = $storage->getQuery();
    $query->accessCheck(FALSE)
          ->sort('id', 'ASC')
          ->range($start, $limit);
    
    $result = $query->execute();

    if (!empty($result)) {
      return $result;
    }

    return NULL;
  }  

  public function getUser($uid) {
    $storage = $this->entityTypeManager->getStorage('user');
    return $storage->load($uid);
  }

  public function getOldCmUserConsentRange($start, $limit) {
    $storage = $this->entityTypeManager->getStorage('cm_user_consent');
    $query = $storage->getQuery();
        
    $query->condition('created', 1731401034, '<')
          ->sort('id')
          ->accessCheck(FALSE)    
          ->range($start, $limit);
    
    $result = $query->execute();

    if (!empty($result)) {

      $entities = $storage->loadMultiple($result);
      return $entities;

    }

    return NULL;

  }



}
