<?php

namespace Drupal\consent_management\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\UserInterface;
use Drupal\consent_management\PolicyInterface;
use Drupal\Core\Render\Markup;

/**
 * Field handler to flag the node type.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("current_policy_state")
 */
class CurrentPolicyState extends FieldPluginBase {

  /**
   * @{inheritdoc}
   */
  public function query() {
    // Leave empty to avoid the field being used in the query.
  }

  /**
   * Define the available options
   * @return array
   */
  protected function defineOptions() {    
    $options = parent::defineOptions();   

    $options['active_policies'] = [
      'default' => 'Bitte auswählen',
    ];

    $options['field_is_empty_text'] = ['default' => ''];

    $options['custom_date_format'] = ['default' => ''];
    
    return $options;
  }  


  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {

    $policy_options = $this->getPolicyOptions();

    $form['active_policies'] = [
      '#type' => 'select',
      '#title' => $this->t('Select an active policy'),
      '#options' => $policy_options,
      '#default_value' =>$this->options['active_policies'], 
    ];   

    $form['field_is_empty_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter text if the field has no value yet'),
      '#default_value' => $this->options['field_is_empty_text'],     
    ];
    
    $form['custom_date_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom date format'),
      '#default_value' => $this->options['custom_date_format'], 
    ];

    parent::buildOptionsForm($form, $form_state);

  }  

  /**
   * @{inheritdoc}
   */
  public function render(ResultRow $values) {


    // Get the entity
    $entity = $this->getEntity($values);

    // Get the selected policy
    $policy_id = $this->options['active_policies'];

    // Get the field you want the output from
    $date_format = $this->options['custom_date_format'];

    // Empty text
    $empty_text = $this->options['field_is_empty_text'];

   
    $policy = \Drupal::entityTypeManager()->getStorage('cm_policy')->load($policy_id);
    if (!$policy instanceof PolicyInterface) {
      return;
    }

    if (!$entity instanceof UserInterface) {
      return;
    }

    $output = $this->getUserConsentStateForPolicy($entity, $policy, $date_format, $empty_text);  

    return $output;

  }

  /**
   * Get active policies
   *
   * @return array
   *   Policies keyed by id.
   */
  protected function getPolicyOptions(): array {

    $options = [];

    $policies = \Drupal::entityTypeManager()->getStorage('cm_policy')->loadMultiple();

    if (!empty($policies)) {
      foreach ($policies as $policy) {
        if ($policy->getStatus() == TRUE) {
          $options[$policy->id()] = $policy->label();
        }        
      }
    }

    return $options;

  }

  /**
   * Get user consent for current policy.
   *
   * @param UserInterface $user
   *   The user entity.
   * @param PolicyInterface $policy
   *   The policy entity.
   * 
   * @return string
   *   The user consent state.  
   */
  protected function getUserConsentStateForPolicy(UserInterface $user, PolicyInterface $policy, $date_format = NULL, $empty_text = NULL) {

    // Per default default no state.
    $output = '-';

    // Build conditions for query    
    $conditions = [
      'uid' => $user->id(),
      'cm_policy_version' => $policy->getPolicyVersionId(),
      'status' => 1,
    ];

    if ($empty_text) {
      $output = $empty_text;
    }

    // Load User consent
    $user_consents = \Drupal::entityTypeManager()->getStorage('cm_user_consent')->loadByProperties($conditions);
    if ($user_consent = reset($user_consents)) {
      // Get the state
      $state = $this->getStateTextFromValue($user_consent->getState());
      // Get the date the consent was given
      if (!$date_format) {
        $date = $user_consent->getCreatedTime();
      } 
      else {
        $date = date($date_format, $user_consent->getCreatedTime());
      }


      $themed_state['#theme'] = 'consent_management_user_consent_state';
      $themed_state['#state'] = $state;
      $themed_state['#created'] = $date;
      

      $output = $themed_state;  
      
    }

    return $output;

  }

  protected function getStateTextFromValue($value) {
    if ($value == 0) {
      $text = $this->t('undecided');
    }
    elseif ($value == 1) {
      $text = $this->t('not agreed');  
    }
    elseif ($value == 2) {
      $text = $this->t('agreed');  
    }

    return $text;

  }

}
