<?php

namespace Drupal\consent_management\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\filter\StringFilter;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\filter\InOperator;
use Drupal\views\ViewExecutable;
use Drupal\views\Views;

/**
 * Filter by consent state.
 *
 * @ingroup views_filter_handlers
 *
 * @ViewsFilter("consent_state_filter")
 */
class ConsentStateFilter extends InOperator {

  /**
   * Flag to indicate undecided user consent.
   */
  const CONSENT_UNSEEN = -1;  

  /**
   * Flag to indicate undecided user consent.
   */
  const CONSENT_UNDECIDED = 0;

  /**
   * Flag to indicate not agreed user consent.
   */
  const CONSENT_NOT_AGREED = 1;

  /**
   * Flag to indicate agreed user consent.
   */
  const CONSENT_AGREED = 2;
 
  
  
  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);
    $this->definition['options callback'] = [$this, 'generateOptions'];
  }


  /**
   * {@inheritdoc}
   */
  public function query() {

    if (!empty($this->value)) {
      $value = $this->value[0];
    }
    else {
      return;
    }  

    $this->ensureMyTable();
    /** @var \Drupal\views\Plugin\views\query\Sql $query */
    $query = $this->query;
    $table = 'cm_user_consent';

    $base_table = $this->view->storage->get('base_table');

    $configuration = [
      'table' => 'cm_user_consent',
      'field' => 'uid',
      'left_table' => $base_table,
      'left_field' => 'uid',
    ];

    $join = Views::pluginManager('join')
      ->createInstance('standard', $configuration);
    
    $alias = $this->query->addRelationship($configuration['table'], $join, $base_table);
    $state = $alias . '.state';
    $cm_policy_version = $alias . '.cm_policy_version';
    $published = $alias . '.status';

    // In order to get a proper result we need to get our active policy versions
    $policy_versions = \Drupal::service('consent_management.helper')->getAllEnabledPolicyVersionIdsAsString();
    
    switch ($value) {
      case self::CONSENT_UNSEEN:
        $this->query->addWhereExpression($this->options['group'], "{$state} IS NULL");
        break;        
      case self::CONSENT_UNDECIDED:
        $this->query->addWhereExpression($this->options['group'], "{$state} = $value AND {$published} = 1 And {$cm_policy_version} IN ($policy_versions)");
        break;
      case self::CONSENT_NOT_AGREED:
        $this->query->addWhereExpression($this->options['group'], "{$state} = $value AND {$published} = 1 And {$cm_policy_version} IN ($policy_versions)");
        break;
      case self::CONSENT_AGREED:
        $this->query->addWhereExpression($this->options['group'], "{$state} = $value AND {$published} = 1 And {$cm_policy_version} IN ($policy_versions)");
        break;
    }  



  }

  /**
   * Retrieves the allowed values for the filter.
   *
   * @return array
   *   An array of allowed values in the form key => label.
   */
  public function generateOptions() {
    return [
      self::CONSENT_UNSEEN => $this->t('Never seen'),
      self::CONSENT_UNDECIDED => $this->t('Undecided'),
      self::CONSENT_NOT_AGREED => $this->t('Not agreed'),
      self::CONSENT_AGREED => $this->t('Agreed'),
    ];
  }  

}
