<?php

declare(strict_types=1);

namespace Drupal\consent_management;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Url;
use Drupal\Core\Link;

/**
 * Provides a list controller for the policy entity type.
 */
final class PolicyListBuilder extends EntityListBuilder {

 

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['label'] = $this->t('Label');
    $header['status'] = $this->t('Status');
    $header['uid'] = $this->t('Author');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    $header['version'] = $this->t('Version');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\consent_management\PolicyInterface $entity */
    $row['id'] = $entity->id();
    $row['label'] = $entity->label();
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $username_options = [
      'label' => 'hidden',
      'settings' => ['link' => $entity->get('uid')->entity->isAuthenticated()],
    ];
    $row['uid']['data'] = $entity->get('uid')->view($username_options);
    $row['created']['data'] = $entity->get('created')->view(['label' => 'hidden']);
    $row['changed']['data'] = $entity->get('changed')->view(['label' => 'hidden']);

    $add_poclicy_versions_link_url = Url::fromRoute("entity.cm_policy_version.add_form", 
    [
      'cm_policy' => $entity->id()
    ],
    [
      'query' => \Drupal::destination()->getAsArray()
    ]);


    if (!$entity->getPolicyVersion()) {
      //Link::fromTextAndUrl($this->t('Add Policy version'), $add_poclicy_versions_link_url)->toRenderable();
      //\Drupal::service('renderer')->render($version_info); 
      $version_info['info'] = [
        '#type' => 'html_tag',
        '#tag' => 'label',
        '#value' => '',
        '#attributes' => ['class' => ['form-item__label', 'form-required']]
      ];
      $version_info = \Drupal::service('renderer')->render($version_info);

    }
    else {
      $version_info = $entity->getPolicyVersion()->label();
    }

    $row['version'] = $version_info;

    return $row + parent::buildRow($entity);

  }

  /**
   * {@inheritdoc}
   *
   * Builds the entity listing as renderable array for table.html.twig.
   *
   * @todo Add a link to add a new item to the #empty text.
   */
  public function render() {

    $this->messenger()->addMessage($this->t("Please note: Only policies with a version will be published! Policies with a missing version will be marked with a 'x'."));
   
    $build = parent::render();  

    return $build;
      
  }  

}
