<?php

declare(strict_types=1);

namespace Drupal\consent_management;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Defines the access control handler for the policy version entity type.
 *
 * phpcs:disable Drupal.Arrays.Array.LongLineDeclaration
 *
 * @see https://www.drupal.org/project/coder/issues/3185082
 */
final class PolicyVersionAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResult {
    if ($account->hasPermission($this->entityType->getAdminPermission())) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    return match($operation) {
      'view' => AccessResult::allowedIfHasPermission($account, 'view cm_policy_version'),
      'update' => AccessResult::allowedIfHasPermission($account, 'edit cm_policy_version'),
      'delete' => AccessResult::allowedIfHasPermission($account, 'delete cm_policy_version'),
      'delete revision' => AccessResult::allowedIfHasPermission($account, 'delete cm_policy_version revision'),
      'view all revisions', 'view revision' => AccessResult::allowedIfHasPermissions($account, ['view cm_policy_version revision', 'view cm_policy_version']),
      'revert' => AccessResult::allowedIfHasPermissions($account, ['revert cm_policy_version revision', 'edit cm_policy_version']),
      default => AccessResult::neutral(),
    };
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL): AccessResult {
    return AccessResult::allowedIfHasPermissions($account, ['create cm_policy_version', 'administer cm_policy_version'], 'OR');
  }

}
