<?php

declare(strict_types=1);

namespace Drupal\consent_management;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\Query\QueryInterface;

/**
 * Provides a list controller for the policy version entity type.
 */
final class PolicyVersionListBuilder extends EntityListBuilder {

  /**
   * Loads entity IDs using a pager sorted by the entity id.
   *
   * @return array
   *   An array of entity IDs.
   */
  protected function getEntityIds() {

    $cm_policy = \Drupal::routeMatch()->getParameter('cm_policy');
    if (!$cm_policy instanceof EntityInterface) {
      $cm_policy = \Drupal::entityTypeManager()->getStorage('cm_policy')->load($cm_policy);
    }    
    
    if(method_exists($this, 'getEntityListQuery')){
      $query = $this->getEntityListQuery();
    }
    else {
      $query = $this->getStorage()->getQuery()
        ->accessCheck(TRUE)
        ->sort($this->entityType->getKey('id'));

      // Only add the pager if a limit is specified.
      if ($this->limit) {
        $query->pager($this->limit);
      }
    }    

    $query->condition('cm_policy', $cm_policy->id());
    
    return $query->execute();

  }  


  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['label'] = $this->t('Label');
    $header['status'] = $this->t('Status');
    $header['uid'] = $this->t('Author');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\consent_management\PolicyVersionInterface $entity */
    $row['id'] = $entity->id();
    $row['label'] = $entity->label();
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $username_options = [
      'label' => 'hidden',
      'settings' => ['link' => $entity->get('uid')->entity->isAuthenticated()],
    ];
    $row['uid']['data'] = $entity->get('uid')->view($username_options);
    $row['created']['data'] = $entity->get('created')->view(['label' => 'hidden']);
    $row['changed']['data'] = $entity->get('changed')->view(['label' => 'hidden']);
    return $row + parent::buildRow($entity);
  }
  
  protected function getEntityListQuery() : QueryInterface {
    $query = $this->getStorage()
        ->getQuery()
        ->accessCheck(TRUE)
        ->sort($this->entityType
        ->getKey('id'));
    // Only add the pager if a limit is specified.
    if ($this->limit) {
        $query->pager($this->limit);
    }
    return $query;
  }  



}
