<?php

namespace Drupal\consumer_base_url;

use Drupal\consumers\Entity\Consumer;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;

/**
 * Consumers base URL provider.
 *
 * @package Drupal\consumer_base_url
 */
class BaseUrlProvider {

  /**
   * Constructs a BaseUrlProvider object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Load the base URL for a given consumer.
   *
   * @param \Drupal\consumers\Entity\Consumer $consumer
   *   Consumer entity to load base URL for.
   *
   * @return string|null
   *   Return the base URL of a consumer as string or NULL.
   */
  public function loadBaseUrl(Consumer $consumer): ?string {
    if (!$consumer->get('base_url')->isEmpty()) {
      return $consumer->get('base_url')->value;
    }
    return NULL;
  }

  /**
   * Check if URL is a consumer route.
   *
   * @param \Drupal\Core\Url $url
   *   The URL object of processed path.
   *
   * @return bool
   *   Return TRUE if URL is a consumer route.
   */
  public function isConsumerRoute(Url $url): bool {
    if ($url->isRouted()
      && in_array($url->getRouteName(), $this->getConsumerRoutes())) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Array of route names handled by consumer.
   *
   * @return string[]
   *   Array of route names handled by consumer.
   *
   * @todo Make consumer routes alterable and/or configurable.
   */
  public function getConsumerRoutes(): array {
    $routes = [
      '<front>',
      'entity.node.canonical',
      'entity.node.latest_version',
      'entity.node.revision',
      'entity.taxonomy_term.canonical',
      'entity.user.canonical',
    ];

    if ($this->configFactory->get('media.settings')->get('standalone_url')) {
      $routes[] = 'entity.media.canonical';
    }

    return $routes;
  }

}
