<?php

namespace Drupal\consumer_base_url\HttpKernel;

use Drupal\Component\Utility\UrlHelper;
use Drupal\consumer_base_url\BaseUrlProvider;
use Drupal\consumers\Negotiator;
use Drupal\Core\PathProcessor\OutboundPathProcessorInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\path_alias\AliasManagerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Processes the outbound path for consumers.
 *
 * This outbound path process will generate correct URLs in backend for viewing
 * of published/unpublished articles in frontend. That's important for editorial
 * workflow.
 */
class ConsumerPathProcessor implements OutboundPathProcessorInterface {

  /**
   * Constructs a ConsumerPathProcessor object.
   *
   * @param \Drupal\consumers\Negotiator $consumerNegotiator
   *   The consumer negotiator.
   * @param \Drupal\consumer_base_url\BaseUrlProvider $baseUrlProvider
   *   The base url provider service.
   * @param \Drupal\path_alias\AliasManagerInterface $aliasManager
   *   The path alias manager.
   */
  public function __construct(
    protected Negotiator $consumerNegotiator,
    protected BaseUrlProvider $baseUrlProvider,
    protected AliasManagerInterface $aliasManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function processOutbound($path, &$options = [], ?Request $request = NULL, ?BubbleableMetadata $bubbleable_metadata = NULL) {
    // Only act on valid internal routed paths and when a domain loads.
    if (empty($path) || !empty($options['external']) || !isset($options['route'])) {
      return $path;
    }

    if (!$options['route']->getOption(ConsumerRouteProcessor::ROUTE_OPTION)) {
      return $path;
    }

    // Workaround for /admin paths without admin route handling.
    if ($path === '/admin' || strpos($path, '/admin/') !== FALSE) {
      return $path;
    }

    if (($consumer = $this->consumerNegotiator->negotiateFromRequest($request))
      && $consumer_base_url = $this->baseUrlProvider->loadBaseUrl($consumer)) {

      // Do not change Base URL in GraphQL 4.x and 3.x requests.
      if (consumer_base_url_is_graphql_request($request)) {
        return $path;
      }

      $options['base_url'] = trim($consumer_base_url, '/');
      $options['absolute'] = TRUE;
      return UrlHelper::encodePath($path);
    }
    return $path;
  }

}
