# Security Policy

## Supported Versions

| Version | Supported |
| ------- |-----------|
| 1.x     | ✔         |

## Reporting a Vulnerability

If you discover a security vulnerability within Contact Block AJAX,
please send an email to the Drupal Security Team at security@drupal.org.

**Do NOT** use the public issue queue for security vulnerabilities.

All security vulnerabilities will be promptly addressed.

## Security Measures

This module implements the following security measures:

### Input Validation
- All user inputs are validated and sanitized
- Form display modes are validated against allowed list
- Wrapper IDs are sanitized using `Html::getId()`
- Contact form machine names validated via routing constraints

### CSRF Protection
- Uses Drupal's native AJAX system with automatic CSRF protection
- All AJAX endpoints validate `_format: ajax` requirement
- Routes require proper authentication via cookie

### XSS Prevention
- All output uses Drupal's render API
- JavaScript uses `textContent` for safe text insertion
- No `innerHTML` with unsanitized content
- Template variables properly escaped via Twig auto-escaping

### Permission Checks
- All routes require `access site-wide contact form` permission
- Rate limit configuration requires `administer form load rate limit` permission
- No permission bypasses in code
- Proper access checking on entities

### Rate Limiting & Flood Protection
- **IP-based rate limiting** using Drupal's Flood API
- **Configurable thresholds**: Prevent abuse with customizable limits
  (1-500 requests)
- **Flexible time windows**: Set protection periods from 1 minute to 24 hours
- **Automatic cleanup**: Expired flood entries are automatically removed
  by Drupal core
- **Detailed logging**: All rate limit violations logged to watchdog
- **User-friendly errors**: Clear messages with retry information
  (RFC 6585 compliance)

#### Rate Limiting Features

The module's rate limiting system provides defense against:
- **Denial of Service (DoS)**: Prevents resource exhaustion from
  excessive requests
- **Scraping attacks**: Limits automated form harvesting
- **Reconnaissance**: Slows down attackers probing for vulnerabilities
- **Resource abuse**: Protects server resources from malicious actors

Default configuration (30 requests per 5 minutes) provides reasonable protection
while allowing legitimate user activity.

#### Recommended Rate Limit Settings

- **Development**: Disable or use high limits (100+ per 5 minutes)
- **Production - Light**: 30 requests per 5 minutes (default)
- **Production - Moderate**: 15 requests per 3 minutes
- **Production - Strict**: 10 requests per 3 minutes

Adjust based on your site's traffic patterns and security requirements.

### Error Handling
- Generic error messages to users (no information disclosure)
- Detailed logging for administrators
- Try-catch blocks around critical operations
- Proper HTTP status codes (400, 403, 429, 500)

### HTTP Security Headers
- `X-Robots-Tag: noindex` prevents indexing of AJAX endpoints
- `Cache-Control: no-cache` prevents sensitive data caching
- Rate limit responses include `Retry-After` header (RFC 6585)

## Dependencies

This module relies on Drupal core's security features:
- Form API for CSRF protection
- AJAX API for secure XHR requests
- Permission system for access control
- Render API for XSS prevention
- Flood API for rate limiting and abuse prevention
- Watchdog for security event logging

## Best Practices

When using this module:

1. **Always use with HTTPS in production** to prevent man-in-the-middle attacks
2. **Keep Drupal core and this module updated** with security patches
3. **Enable rate limiting** to prevent abuse (recommended: 30 per 5 minutes
   minimum)
4. **Use anti-spam modules** (CAPTCHA, reCAPTCHA, Honeypot) for form submission
   protection
5. **Monitor logs regularly** for suspicious activity:

```bash
drush wd-show --type=contact_block_ajax
```
6. Review flood entries periodically:

```bash
drush sqlq "SELECT identifier, COUNT(*) as count FROM flood \
WHERE event = 'contact_block_ajax.form_load' GROUP BY identifier \
ORDER BY count DESC LIMIT 10"
```
7. Configure appropriate permissions - restrict administer form load
   rate limit to trusted admins
8. Follow Drupal security best practices -
   https://www.drupal.org/security/secure-configuration

## Security Considerations

### Rate Limiting Bypass Attempts

Be aware that attackers may attempt to bypass rate limiting:

- **IP rotation**: Attackers may use multiple IPs or proxies
  - **Mitigation**: Consider user session-based limits (requires custom code)
  - **Mitigation**: Use CDN-level rate limiting (Cloudflare, AWS WAF)

- **Distributed attacks**: Attackers may use botnets
  - **Mitigation**: Implement additional layers (Web Application Firewall)
  - **Mitigation**: Monitor for patterns in watchdog logs

- **Legitimate shared IPs**: Corporate networks, schools may share IPs
  - **Mitigation**: Set reasonable limits to avoid blocking legitimate users
  - **Mitigation**: Provide clear error messages with retry information

### Flood Table Growth

The flood table may grow large on high-traffic sites:

- Drupal core automatically cleans up expired entries via cron
- Monitor table size: `drush sqlq "SELECT COUNT(*) FROM flood WHERE event =
  'contact_block_ajax.form_load'"`
- Manual cleanup if needed: `drush sqlq "DELETE FROM flood WHERE event =
  'contact_block_ajax.form_load'"`
- Use [Flood control](https://www.drupal.org/project/flood_control) module
  to manage the flood table.
  - Manage the flood table at People > Flood Unblock
    (`/admin/people/flood-unblock`)

### Information Disclosure

Rate limit error messages intentionally avoid disclosing:

- Exact threshold values (prevents optimization of attacks)
- Internal configuration details
- Server resource information

However, the Retry-After header reveals the time window, which is acceptable
per RFC 6585 and aids legitimate users.

## Known Security Limitations

1. Personal contact forms: Not supported - only site-wide forms
2. Rate limiting scope: Per-IP only, not per-user or per-session
3. No CAPTCHA for loading: Forms can be loaded without CAPTCHA
   (use anti-spam modules for submission)
4. JavaScript requirement: Users without JavaScript see fallback message
   (by design)

## Security Hardening Recommendations

For additional security:

1. Implement WAF rules at the web server or CDN level
2. Use fail2ban to ban IPs with excessive 429 errors
3. Enable Drupal's session-based flood limits for form submissions
4. Monitor for patterns in watchdog logs indicating coordinated attacks
5. Consider additional authentication for sensitive contact forms
6. Implement honeypot fields to catch bots

## Compliance

This module is designed to be compatible with:

- OWASP Top 10 guidelines
- Drupal Coding Standards security requirements
- GDPR: Does not store personally identifiable information (only IPs in flood
  table, which is standard Drupal behavior)
- RFC 6585: HTTP 429 Too Many Requests response

## Security Advisories

Security advisories for this module will be published at:
https://www.drupal.org/project/contact_block_ajax/security

Subscribe to security announcements:
https://www.drupal.org/security
