/**
 * @file
 * Contact Block Ajax behaviors.
 */
(function ($, Drupal, once) {

  'use strict';

  const CONTACT_BLOCK_WRAPPER = '.contact-block-ajax-wrapper';
  const CONTACT_FORM_CONTAINER = '.ajax-contact-form-container';

  /**
   * Attach contact block ajax behavior.
   */
  Drupal.behaviors.contactBlockAjax = {
    attach(context, settings) {
      const containers = once('contact-block-ajax', CONTACT_BLOCK_WRAPPER, context);

      containers.forEach(function (container) {
        const config = settings.contactBlockAjax || {};
        const placeholder = container.querySelector(CONTACT_FORM_CONTAINER);

        if (!placeholder || placeholder.dataset.loaded === 'true') {
          return;
        }
        config.wrapperSelector = `#${placeholder.id}`;

        // Initialize Intersection Observer.
        if ('IntersectionObserver' in window) {
          const observer = new IntersectionObserver(
            function (entries) {
              entries.forEach(function (entry) {
                if (entry.isIntersecting) {
                  loadContactForm(placeholder, config);
                  observer.unobserve(entry.target);
                }
              });
            },
            {
              root: null,
              rootMargin: config.rootMargin || '50px 0px',
              threshold: config.threshold || 0.1
            }
          );
          observer.observe(container);

          // Store observer reference for cleanup.
          container.intersectionObserver = observer;
        }
        else {
          // Fallback for older browsers - load immediately.
          loadContactForm(placeholder, config);
        }
      });
    },

    detach: function (context, settings, trigger) {
      if (trigger === 'unload') {
        const containers = context.querySelectorAll(CONTACT_BLOCK_WRAPPER);
        containers.forEach(function (container) {
          if (container.intersectionObserver) {
            container.intersectionObserver.disconnect();
            delete container.intersectionObserver;
          }
        });
      }
    }
  };

  /**
   * Load contact form via AJAX.
   *
   * @param {HTMLElement} placeholder
   *   The placeholder element.
   * @param {object} config
   *   Configuration object.
   */
  function loadContactForm(placeholder, config) {
    if (!placeholder || placeholder.dataset.loaded === 'true') {
      return;
    }

    // Mark as loading to prevent duplicate requests.
    placeholder.dataset.loaded = 'true';
    placeholder.setAttribute('aria-busy', 'true');
    showLoadingSpinner(placeholder);

    // Get AJAX URL from data attribute.
    const ajaxUrl = placeholder.dataset.ajaxUrl;
    if (!ajaxUrl) {
      showErrorMessage(
        placeholder,
        Drupal.t('Configuration error: No AJAX URL provided.')
      );
      return;
    }

    // Create AJAX request using Drupal's AJAX API.
    const ajax = Drupal.ajax({
      url: ajaxUrl,
      base: placeholder.id,
      element: placeholder,
      progress: {
        type: 'none' // We handle our own loading indicator.
      }
    });

    // Override error handler for better UX.
    ajax.error = function (xhr, uri, customMessage) {
      hideLoadingSpinner(placeholder);
      placeholder.setAttribute('aria-busy', 'false');

      let errorMessage = Drupal.t('Failed to load contact form.');

      // Provide specific error messages based on HTTP status.
      if (xhr.status === 403) {
        errorMessage = Drupal.t('You do not have permission to access this form.');
      }
      else if (xhr.status === 404) {
        errorMessage = Drupal.t('The requested contact form was not found.');
      }
      else if (xhr.status === 400) {
        errorMessage = Drupal.t('Invalid request. Please refresh the page and try again.');
      }
      else if (xhr.responseJSON && xhr.responseJSON.message) {
        errorMessage = xhr.responseJSON.message;
      }
      else if (customMessage) {
        errorMessage = customMessage;
      }

      showErrorMessage(placeholder, errorMessage);

      // Log error for debugging (only in development).
      if (console && console.error) {
        console.error('Contact Block AJAX error:', {
          status: xhr.status,
          statusText: xhr.statusText,
          url: uri
        });
      }
    };

    // Execute AJAX request.
    try {
      ajax.execute();
    }
    catch (error) {
      hideLoadingSpinner(placeholder);
      placeholder.setAttribute('aria-busy', 'false');
      showErrorMessage(
        placeholder,
        Drupal.t('An error occurred while loading the form.')
      );

      if (console && console.error) {
        console.error('Contact Block AJAX exception:', error);
      }
    }
  }

  /**
   * Show loading spinner.
   *
   * @param {HTMLElement} element
   *   The element to show spinner in.
   */
  function showLoadingSpinner(element) {
    const spinner = document.createElement('div');
    spinner.className = 'contact-ajax-spinner';
    spinner.setAttribute('role', 'status');
    spinner.setAttribute('aria-live', 'polite');

    const container = document.createElement('div');
    container.className = 'spinner-container';

    const circle = document.createElement('div');
    circle.className = 'spinner-circle';
    circle.setAttribute('aria-hidden', 'true');

    const text = document.createElement('div');
    text.className = 'spinner-text';
    text.textContent = Drupal.t('Loading contact form...');

    container.appendChild(circle);
    container.appendChild(text);
    spinner.appendChild(container);

    element.innerHTML = '';
    element.appendChild(spinner);
    element.classList.add('is-loading');
  }

  /**
   * Hide loading spinner.
   *
   * @param {HTMLElement} element
   *   The element containing the spinner.
   */
  function hideLoadingSpinner(element) {
    const spinner = element.querySelector('.contact-ajax-spinner');
    if (spinner) {
      spinner.remove();
    }
    element.classList.remove('is-loading');
  }

  /**
   * Show error message.
   *
   * @param {HTMLElement} element
   *   The element to show error in.
   * @param {string} message
   *   The error message to display.
   */
  function showErrorMessage(element, message) {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'contact-ajax-error';
    errorDiv.setAttribute('role', 'alert');
    errorDiv.setAttribute('aria-live', 'assertive');

    const errorContent = document.createElement('div');
    errorContent.className = 'error-content';

    const errorIcon = document.createElement('span');
    errorIcon.className = 'error-icon';
    errorIcon.setAttribute('aria-hidden', 'true');
    errorIcon.textContent = '⚠️';

    const errorMessage = document.createElement('div');
    errorMessage.className = 'error-message';
    errorMessage.textContent = message;

    errorContent.appendChild(errorIcon);
    errorContent.appendChild(errorMessage);
    errorDiv.appendChild(errorContent);

    element.innerHTML = '';
    element.appendChild(errorDiv);
    element.classList.add('has-error');
  }

})(jQuery, Drupal, once);
