<?php

namespace Drupal\contact_block_ajax\Controller;

use Drupal\Component\Utility\Html;
use Drupal\contact\ContactFormInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Flood\FloodInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException;

/**
 * Returns responses for Contact Block AJAX routes.
 */
class ContactBlockAjaxController extends ControllerBase {

  /**
   * The flood event name for form loading.
   */
  const FLOOD_EVENT = 'contact_block_ajax.form_load';

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The entity form builder.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected $entityFormBuilder;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The flood control mechanism.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * Constructs a ContactBlockAjaxController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood control mechanism.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $entity_display_repository,
    EntityFormBuilderInterface $entity_form_builder,
    LoggerInterface $logger,
    FloodInterface $flood,
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityDisplayRepository = $entity_display_repository;
    $this->entityFormBuilder = $entity_form_builder;
    $this->logger = $logger;
    $this->flood = $flood;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('entity.form_builder'),
      $container->get('logger.factory')->get('contact_block_ajax'),
      $container->get('flood')
    );
  }

  /**
   * Loads and returns a contact form via AJAX.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Drupal\contact\ContactFormInterface|null $contact_form
   *   The contact form entity.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response containing the form.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Thrown when request validation fails.
   */
  public function loadForm(Request $request, ?ContactFormInterface $contact_form = NULL): AjaxResponse {
    // Apply rate limiting if enabled.
    $this->checkRateLimit($request);

    // Validate AJAX request.
    if (!$request->isXmlHttpRequest()) {
      throw new BadRequestHttpException('This endpoint only accepts AJAX requests.');
    }

    // Validate contact form exists.
    if (!$contact_form) {
      throw new BadRequestHttpException('Contact form not found.');
    }

    // Validate wrapper_id parameter.
    $block_wrapper_id = $request->query->get('wrapper_id');
    if (empty($block_wrapper_id) || !is_string($block_wrapper_id)) {
      throw new BadRequestHttpException('Missing or invalid wrapper_id parameter.');
    }

    // Sanitize wrapper_id to prevent XSS.
    $block_wrapper_id = Html::getId($block_wrapper_id);
    $block_selector = '#' . $block_wrapper_id . '.ajax-contact-form-container';
    $form_wrapper_id = $block_wrapper_id . '-form';

    // Validate and sanitize form_display parameter.
    $form_display = $request->query->get('display', 'default');
    if (!is_string($form_display)) {
      throw new BadRequestHttpException('Invalid display parameter.');
    }

    // Validate form display exists.
    $available_displays = $this->entityDisplayRepository->getFormModeOptions('contact_message');
    if (!isset($available_displays[$form_display])) {
      $this->logger->warning(
        'Invalid form display "@display" requested for contact form @form',
        [
          '@display' => $form_display,
          '@form' => $contact_form ? $contact_form->id() : 'unknown',
        ]
      );
      throw new BadRequestHttpException('Invalid form display.');
    }

    // Create contact message entity.
    $message = $this->entityTypeManager
      ->getStorage('contact_message')
      ->create([
        'contact_form' => $contact_form->id(),
      ]);

    // Build the form.
    $form = $this->entityFormBuilder->getForm($message, $form_display, [
      'contact_block_ajax_form' => TRUE,
      'wrapper_id' => $form_wrapper_id,
    ]);
    $form['#form_display'] = $form_display;

    // Wrap form in container.
    $form_container = [
      'contact_wrapper' => [
        '#type' => 'container',
        '#attributes' => [
          'id' => $form_wrapper_id,
          'class' => [Html::getClass($form_wrapper_id)],
        ],
        'form' => $form,
      ],
    ];

    // Create AJAX response.
    $response = new AjaxResponse();
    $response->addCommand(new HtmlCommand($block_selector, $form_container));
    $response->addCommand(new InvokeCommand($block_selector, 'removeClass', ['is-loading']));
    $response->addCommand(new InvokeCommand($block_selector, 'addClass', ['form-loaded']));
    $response->addCommand(new InvokeCommand($block_selector, 'attr', ['aria-busy', 'false']));

    // Set cache headers.
    $response->setMaxAge(0);
    $response->headers->set('X-Robots-Tag', 'noindex');

    return $response;
  }

  /**
   * Checks if the request exceeds the rate limit.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException
   *   Thrown when rate limit is exceeded.
   */
  protected function checkRateLimit(Request $request): void {
    // Load rate limit configuration (read-only).
    $rate_limit_config = $this->configFactory
      ->get('contact_block_ajax.form_load_rate_limit');

    // Skip if rate limiting is disabled.
    if (!$rate_limit_config->get('enabled')) {
      return;
    }

    // Get rate limit parameters from configuration.
    $threshold = (int) $rate_limit_config->get('limit');
    $window = (int) $rate_limit_config->get('interval');
    $identifier = $request->getClientIp();

    // Check if request exceeds the allowed rate.
    if (!$this->flood->isAllowed(self::FLOOD_EVENT, $threshold, $window, $identifier)) {
      // Log the rate limit violation.
      $this->logger->warning(
        'Rate limit exceeded for contact form loading. IP: @ip, Limit: @limit/@window seconds',
        [
          '@ip' => $identifier,
          '@limit' => $threshold,
          '@window' => $window,
        ]
      );

      // Throw exception with retry-after header.
      throw new TooManyRequestsHttpException(
        $window,
        sprintf('Too many form load requests. Please wait %d seconds before trying again.', $window),
      );
    }

    // Register this request in the flood table.
    $this->flood->register(self::FLOOD_EVENT, $window, $identifier);
  }

}
