<?php

namespace Drupal\contact_block_ajax\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Contact Block AJAX form load rate limit settings.
 */
class RateLimitForm extends ConfigFormBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Constructs a RateLimitForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    DateFormatterInterface $date_formatter,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'contact_block_ajax_form_load_rate_limit';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['contact_block_ajax.form_load_rate_limit'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('contact_block_ajax.form_load_rate_limit');

    $form['rate_limit'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Form Load Rate Limiting'),
      '#description' => $this->t('Prevent abuse by limiting how often contact forms can be loaded via AJAX.'),
    ];

    $form['rate_limit']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rate limiting'),
      '#description' => $this->t('When enabled, limits the number of times a user can load contact forms within a specified time window.'),
      '#default_value' => $config->get('enabled'),
    ];

    $options = $this->getOptions();

    $form['rate_limit']['limit'] = [
      '#type' => 'select',
      '#title' => $this->t('Maximum requests'),
      '#description' => $this->t('The maximum number of form loads allowed within the time window.'),
      '#options' => array_combine($options['counter'], $options['counter']),
      '#default_value' => $config->get('limit'),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['rate_limit']['interval'] = [
      '#type' => 'select',
      '#title' => $this->t('Time window'),
      '#description' => $this->t('The time period for measuring the request limit.'),
      '#options' => array_map(
        [$this->dateFormatter, 'formatInterval'],
        array_combine($options['time'], $options['time'])
      ),
      '#default_value' => $config->get('interval'),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['rate_limit']['example'] = [
      '#type' => 'item',
      '#markup' => $this->t('<strong>Example:</strong> With a limit of @limit requests per @interval, each IP address can load contact forms up to @limit times within any @interval period.', [
        '@limit' => $config->get('limit'),
        '@interval' => $this->dateFormatter->formatInterval($config->get('interval')),
      ]),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('contact_block_ajax.form_load_rate_limit')
      ->set('enabled', (bool) $form_state->getValue('enabled'))
      ->set('limit', (int) $form_state->getValue('limit'))
      ->set('interval', (int) $form_state->getValue('interval'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Provides options for the rate limit configuration.
   *
   * @return array
   *   An array with 'counter' and 'time' keys containing available options.
   */
  protected function getOptions(): array {
    return [
      'counter' => [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 20, 30, 40, 50, 75, 100, 125, 150, 200, 250, 500],
      'time' => [60, 180, 300, 600, 900, 1800, 2700, 3600, 10800, 21600, 32400, 43200, 86400],
    ];
  }

}
