# Contact Block AJAX

## INTRODUCTION

The Contact Block AJAX module provides blocks that load Drupal contact forms via
AJAX with lazy loading using the Intersection Observer API.
This improves page load performance by deferring form rendering until the block
enters the viewport.

**Key Features:**
- **Performance optimization**: Defers loading of contact forms until visible
- **Enhanced UX**: Submit forms via AJAX without page reloads
- **Personal contact forms**: Full support for user-specific contact forms
- **Rate limiting**: Configurable protection against abuse
  and excessive requests
- **Flexible configuration**: Support for multiple contact forms
  and display modes
- **Security integration**: Works with CAPTCHA, reCAPTCHA, and Honeypot modules
- **Compatibility**: Works alongside contact_ajax module (with limitations)

## REQUIREMENTS

**Required modules:**
- Block (core)
- Contact (core)

**Recommended modules:**
- [CAPTCHA](https://www.drupal.org/project/captcha) - Add CAPTCHA protection
- [reCAPTCHA](https://www.drupal.org/project/recaptcha) - Add Google reCAPTCHA
- [Honeypot](https://www.drupal.org/project/honeypot) - Spam prevention
- [Contact AJAX](https://www.drupal.org/project/contact_ajax) -
  Enhanced AJAX features
- [Flood control](https://www.drupal.org/project/flood_control) -
  Interface for hidden flood control variables

## INSTALLATION

Install as you would normally install a contributed Drupal module.
Visit https://www.drupal.org/node/1897420 for further information.

```bash
composer require drupal/contact_block_ajax
drush en contact_block_ajax -y
```

## CONFIGURATION

### Basic Setup

1. Navigate to **Structure > Block layout** (`/admin/structure/block`)
2. Click **"Place block"** in your desired region
3. Find **"Contact Block AJAX"** and click **"Place block"**
4. Configure the block:
   - **Contact form**: Select which contact form to display
   - **Form display mode**: Choose display mode (default or custom)
   - **Wrapper ID**: Enter a unique wrapper ID
5. Save the block configuration

The form will now lazy-load when the block enters the viewport.

### Rate Limiting Configuration

Configure rate limiting to prevent abuse and excessive form loading:

1. Navigate to Configuration > People > Form load rate limit
   `(/admin/config/people/contact-block-ajax-rate-limit)`
2. Enable rate limiting with the checkbox
3. Configure the settings:
   - **Maximum requests**: Number of form loads allowed (1-500)
   - **Time window**: Time period for measuring limits (1 minute - 24 hours)
4. Save the configuration

#### Example configurations:

- **Light protection**: 30 requests per 5 minutes
- **Development**: Disable rate limiting for testing

#### Rate limiting features:

- IP-based tracking using Drupal's Flood API
- Configurable thresholds and time windows
- Automatic cleanup of expired entries
- Detailed logging of rate limit violations
- User-friendly error messages with retry information

#### Rate limiting cleanup:
Use [Flood control](https://www.drupal.org/project/flood_control) module
to manage the flood table.
- Manage the flood table at People > Flood Unblock
  (`/admin/people/flood-unblock`)

### Advanced Configuration

#### Custom Form Display Modes

Create custom form display modes at:
**Structure > Contact forms > [Your Form] > Manage form display**
(`/admin/structure/contact/manage/[form-id]/form-display`)

#### Theming

Override the template in your theme:

```
your_theme/
  templates/
    contact-block-ajax.html.twig
```

Available variables:
- `form_id`: Contact form machine name
- `form_display`: Display mode being used
- `wrapper_id`: Unique wrapper ID
- `ajax_url`: URL for AJAX loading
- `attributes`: HTML attributes

#### JavaScript Settings

Adjust intersection observer behavior in your theme:

```javascript
drupalSettings.contactBlockAjax = {
  threshold: 0.1,        // Visibility threshold (0-1)
  rootMargin: '50px 0px' // Load margin around viewport
};
```

## COMPATIBILITY & INTEGRATION

### Contact AJAX Module

This module is **compatible** with
[Contact AJAX](https://www.drupal.org/project/contact_ajax).

When both modules are enabled:
- Contact AJAX settings take precedence for form submission
  if "Ajax Form" option is enabled.
- Form loading is handled by Contact Block AJAX.
- **IMPORTANT**: The "Default message and empty form"
  confirmation type in contact_ajax is **NOT compatible** with this module.
  Use "Default message", "Node content" or "Custom message" instead.

### Anti-Spam Modules

Works seamlessly with:
- **[CAPTCHA](https://www.drupal.org/project/captcha)** /
  **[Image CAPTCHA](https://www.drupal.org/project/captcha)**:
  Challenge-response protection
- **[reCAPTCHA](https://www.drupal.org/project/recaptcha)**:
  Google reCAPTCHA v2/v3
- **[Honeypot](https://www.drupal.org/project/honeypot)**:
  Invisible spam prevention
- **[Flood control](https://www.drupal.org/project/flood_control)**:
  Interface for hidden flood control variables

## PERFORMANCE BENEFITS

### Before (Standard Contact Block)
- Form HTML loaded on initial page load
- Increases page weight: ~50-100KB
- Longer Time to Interactive (TTI)

### After (Contact Block AJAX)
- Form loaded only when visible
- Reduced initial page weight
- Faster initial page load
- Better Core Web Vitals scores

### Typical Improvements
- **Initial page size**: 15-30% reduction
- **Time to Interactive**: 200-500ms faster
- **Lighthouse score**: +5-10 points

## TROUBLESHOOTING

### Form Not Loading

**Symptoms**: Spinner shows indefinitely or no form appears

**Solutions**:
1. Check JavaScript console for errors (F12 in browser)
2. Verify contact form permissions at **People > Permissions**
3. Clear Drupal cache: `drush cr`
4. Ensure JavaScript is enabled in browser
5. Check that contact form exists and is not disabled
6. For personal forms, verify user has enabled personal contact form

### Form Submission Not Working

**Symptoms**: Form submits but doesn't show confirmation

**Solutions**:
1. Clear all caches: `drush cr`
2. Check AJAX errors in browser console
3. Verify form permissions
4. Disable contact_ajax temporarily to isolate issue
5. Check server error logs

### Personal Contact Form Issues

**Symptoms**: Personal contact form not showing on user pages

**Solutions**:
1. Verify personal contact forms are enabled globally
2. Check that the user has enabled their personal contact form
3. Ensure block visibility is set to show on `/user/*` pages
4. Verify you have permission to use personal contact form
5. Check browser console for specific error messages

### Rate Limit Issues
**Symptoms**: "Too many requests" error appearing

**Solutions**:

1. Check rate limit configuration at
   `/admin/config/people/contact-block-ajax-rate-limit`
2. Verify the time window hasn't passed yet
3. Clear flood entries: `drush sqlq "DELETE FROM flood WHERE
   event = 'contact_block_ajax.form_load'"`
4. Temporarily disable rate limiting for testing
5. Check watchdog logs: `drush wd-show --type=contact_block_ajax`

**Debugging rate limits**:

```bash
# View current flood entries
drush sqlq "SELECT * FROM flood WHERE event = 'contact_block_ajax.form_load'"

# Clear specific IP
drush sqlq "DELETE FROM flood WHERE event = 'contact_block_ajax.form_load' \
AND identifier = '192.168.1.1'"
```

### Compatibility Issues with Contact AJAX

If using contact_ajax module:
1. Disable "**Default message and empty form**" confirmation type
2. Use "Default message", "Node content" or "Custom message" instead
3. Clear cache after changing settings

### Performance Issues

If forms load slowly:
1. Check network tab in browser DevTools
2. Ensure CDN/caching is properly configured
3. Consider using Drupal's BigPipe module
4. Optimize contact form field configuration

### Browser Compatibility

**Supported**: Modern browsers with IntersectionObserver API
- Chrome 51+
- Firefox 55+
- Safari 12.1+
- Edge 15+

**Fallback**: Older browsers load form immediately (no lazy loading)

## SECURITY

This module follows Drupal security best practices:

- CSRF protection via Drupal's AJAX system
- XSS prevention through proper sanitization
- Permission checking on all operations
- Input validation on all parameters
- IP-based rate limiting to prevent abuse
- Flood API integration for automatic cleanup
- Generic error messages to prevent information disclosure
- Secure handling of personal contact forms with user validation

Report security issues to the Drupal Security Team:
https://www.drupal.org/drupal-security-team

### PERMISSIONS

- **access site-wide contact form**: Required to load and submit contact forms
- **administer form load rate limit**: Allows configuration of rate
  limiting settings
- **use personal contact forms**: Required for personal
  contact forms (core permission)

## DEVELOPMENT & TESTING

### Running Tests

```bash
# Run specific test class
phpunit modules/contrib/contact_block_ajax/tests/src/Functional/ContactBlockAjaxTest.php
```

### Code Standards

```bash
# Check coding standards
phpcs --standard=Drupal,DrupalPractice modules/contrib/contact_block_ajax

# Fix coding standards
phpcbf --standard=Drupal,DrupalPractice modules/contrib/contact_block_ajax
```

## MAINTAINERS

- Alexandru Stefan (stf54) - https://www.drupal.org/u/stf54

## RELATED MODULES

- **[Contact AJAX](https://www.drupal.org/project/contact_ajax)**:
  Enhanced AJAX functionality
- **[Contact Block](https://www.drupal.org/project/contact_block)**:
  Provides contact forms in a block

## CONTRIBUTING

Contributions are welcome! Please:
1. Create issues in the issue queue
2. Follow Drupal coding standards
3. Add tests for new features
4. Update documentation

Issue queue: https://www.drupal.org/project/issues/contact_block_ajax
