/**
 * @file
 * Contact Block Ajax behaviors.
 *
 * Handles lazy loading of contact forms using Intersection Observer API
 * with graceful fallbacks for older browsers.
 */
(function ($, Drupal, once) {

  'use strict';

  // Constants for selectors and configuration.
  const CONTACT_BLOCK_WRAPPER = '.contact-block-ajax-wrapper';
  const CONTACT_FORM_CONTAINER = '.ajax-contact-form-container';
  const DEFAULT_ROOT_MARGIN = '50px 0px';
  const DEFAULT_THRESHOLD = 0.1;

  /**
   * Attach contact block ajax behavior.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches lazy loading behavior to contact form blocks.
   * @prop {Drupal~behaviorDetach} detach
   *   Cleans up observers when elements are removed.
   */
  Drupal.behaviors.contactBlockAjax = {
    /**
     * Attach the lazy loading behavior to contact blocks.
     *
     * @param {Element} context
     *   The context element.
     * @param {Object} settings
     *   The drupal settings object.
     */
    attach(context, settings) {
      const containers = once('contact-block-ajax', CONTACT_BLOCK_WRAPPER, context);

      containers.forEach(
        function (container) {
          const config = settings.contactBlockAjax || {};
          const placeholder = container.querySelector(CONTACT_FORM_CONTAINER);

          // Skip if placeholder doesn't exist or is already loaded.
          if (!placeholder || placeholder.dataset.loaded === 'true') {
            return;
          }

          config.wrapperSelector = `#${placeholder.id}`;

          // Use Intersection Observer for efficient lazy loading.
          if ('IntersectionObserver' in window) {
            initIntersectionObserver(container, placeholder, config);
          } else {
            // Fallback for older browsers - load immediately.
            loadContactForm(placeholder, config);
          }
        },
      );
    },

    /**
     * Detach the behavior and clean up observers.
     *
     * @param {Element} context
     *   The context element.
     * @param {Object} settings
     *   The drupal settings object.
     * @param {string} trigger
     *   The trigger for the detach ('unload', etc.).
     */
    detach: function (context, settings, trigger) {
      if (trigger === 'unload') {
        const containers = context.querySelectorAll(CONTACT_BLOCK_WRAPPER);
        containers.forEach(
          function (container) {
            if (container.intersectionObserver) {
              container.intersectionObserver.disconnect();
              delete container.intersectionObserver;
            }
          },
        );
      }
    },
  };

  /**
   * Initialize Intersection Observer for lazy loading.
   *
   * @param {HTMLElement} container
   *   The container element.
   * @param {HTMLElement} placeholder
   *   The placeholder element.
   * @param {Object} config
   *   Configuration object.
   */
  function initIntersectionObserver(container, placeholder, config) {
    const observer = new IntersectionObserver(
      function (entries) {
        entries.forEach(
          function (entry) {
            if (entry.isIntersecting) {
              loadContactForm(placeholder, config);
              observer.unobserve(entry.target);
            }
          },
        );
      },
      {
        root: null,
        rootMargin: config.rootMargin || DEFAULT_ROOT_MARGIN,
        threshold: config.threshold || DEFAULT_THRESHOLD,
      },
    );

    observer.observe(container);

    // Store observer reference for cleanup.
    container.intersectionObserver = observer;
  }

  /**
   * Load contact form via AJAX.
   *
   * @param {HTMLElement} placeholder
   *   The placeholder element.
   * @param {Object} config
   *   Configuration object.
   */
  function loadContactForm(placeholder, config) {
    // Prevent duplicate requests.
    if (!placeholder || placeholder.dataset.loaded === 'true') {
      return;
    }

    // Mark as loading.
    placeholder.dataset.loaded = 'true';
    placeholder.setAttribute('aria-busy', 'true');
    showLoadingSpinner(placeholder);

    // Validate configuration.
    const ajaxUrl = placeholder.dataset.ajaxUrl;
    if (!ajaxUrl) {
      handleError(
        placeholder,
        Drupal.t('Configuration error. Please contact the site administrator.'),
        'Configuration error: No AJAX URL provided',
      );
      return;
    }

    // Create AJAX request using Drupal's AJAX API.
    const ajax = Drupal.ajax(
      {
        url: ajaxUrl,
        base: placeholder.id,
        element: placeholder,
        progress: {
          type: 'none', // We handle our own loading indicator.
        },
      },
    );

    // Override error handler for better security and UX.
    ajax.error = function (xhr, uri, customMessage) {
      handleAjaxError(placeholder, xhr, uri, customMessage);
    };

    // Execute AJAX request with error handling.
    try {
      ajax.execute();
    } catch (error) {
      handleError(
        placeholder,
        Drupal.t('Unable to load the contact form. Please try again later.'),
        'Contact Block AJAX exception: ' + error,
      );
    }
  }

  /**
   * Handle AJAX errors with appropriate user messaging.
   *
   * Security: We use generic messages for end users to prevent information
   * leakage about form existence, permissions, or system configuration.
   * Detailed error information is only logged to console in development mode.
   *
   * @param {HTMLElement} placeholder
   *   The placeholder element.
   * @param {XMLHttpRequest} xhr
   *   The XHR object.
   * @param {string} uri
   *   The request URI.
   * @param {string} customMessage
   *   Any custom error message.
   */
  function handleAjaxError(placeholder, xhr, uri, customMessage) {
    hideLoadingSpinner(placeholder);
    placeholder.setAttribute('aria-busy', 'false');

    // Default generic error message for security.
    let userMessage = Drupal.t('Unable to load the contact form. Please try again later.');

    // Only provide specific messages for rate limiting (safe to expose).
    if (xhr.status === 429) {
      // Try to extract retry time from response.
      const retryAfter = xhr.getResponseHeader('Retry-After');
      if (retryAfter) {
        userMessage = Drupal.t(
          'Too many requests. Please wait @seconds seconds before trying again.', {
            '@seconds': retryAfter,
          },
        );
      } else {
        userMessage = Drupal.t('Too many requests. Please wait a moment before trying again.');
      }
    } else if (xhr.responseJSON && xhr.responseJSON.message && xhr.status === 429) {
      // Use server-provided message for rate limiting.
      userMessage = xhr.responseJSON.message;
    }

    // Log detailed error for developers/debugging (only in development).
    const debugInfo = {
      status: xhr.status,
      statusText: xhr.statusText,
      url: uri,
      customMessage: customMessage,
      response: xhr.responseText ? xhr.responseText.substring(0, 200) : NULL,
    };

    handleError(placeholder, userMessage, 'AJAX error: ' + JSON.stringify(debugInfo));
  }

  /**
   * Handle errors consistently.
   *
   * @param {HTMLElement} placeholder
   *   The placeholder element.
   * @param {string} userMessage
   *   The message to display to the user.
   * @param {string} logMessage
   *   The message to log for debugging.
   */
  function handleError(placeholder, userMessage, logMessage) {
    hideLoadingSpinner(placeholder);
    placeholder.setAttribute('aria-busy', 'false');
    showErrorMessage(placeholder, userMessage);

    // Only log in development mode or if console exists.
    if (console && console.error && isDevelopmentMode()) {
      console.error('Contact Block AJAX:', logMessage);
    }
  }

  /**
   * Check if we're in development mode.
   *
   * @return {boolean}
   *   True if in development mode.
   */
  function isDevelopmentMode() {
    // Check for Drupal development indicators.
    return (
      typeof drupalSettings !== 'undefined' &&
      drupalSettings.ajaxPageState &&
      drupalSettings.ajaxPageState.theme_token &&
      window.location.hostname === 'localhost' ||
      window.location.hostname.includes('.local') ||
      window.location.hostname.includes('.test') ||
      window.location.hostname.includes('.ddev.site')
    );
  }

  /**
   * Show loading spinner.
   *
   * @param {HTMLElement} element
   *   The element to show spinner in.
   */
  function showLoadingSpinner(element) {
    const spinner = document.createElement('div');
    spinner.className = 'contact-ajax-spinner';
    spinner.setAttribute('role', 'status');
    spinner.setAttribute('aria-live', 'polite');

    const container = document.createElement('div');
    container.className = 'spinner-container';

    const circle = document.createElement('div');
    circle.className = 'spinner-circle';
    circle.setAttribute('aria-hidden', 'true');

    const text = document.createElement('div');
    text.className = 'spinner-text visually-hidden';
    text.setAttribute('aria-hidden', 'true');
    text.textContent = Drupal.t('Loading contact form...');

    const visualText = document.createElement('div');
    visualText.className = 'spinner-visual-text';
    visualText.textContent = Drupal.t('Loading...');

    container.appendChild(circle);
    container.appendChild(text);
    container.appendChild(visualText);
    spinner.appendChild(container);

    element.innerHTML = '';
    element.appendChild(spinner);
    element.classList.add('is-loading');
  }

  /**
   * Hide loading spinner.
   *
   * @param {HTMLElement} element
   *   The element containing the spinner.
   */
  function hideLoadingSpinner(element) {
    const spinner = element.querySelector('.contact-ajax-spinner');
    if (spinner) {
      spinner.remove();
    }
    element.classList.remove('is-loading');
  }

  /**
   * Show error message.
   *
   * @param {HTMLElement} element
   *   The element to show error in.
   * @param {string} message
   *   The error message to display.
   */
  function showErrorMessage(element, message) {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'messages messages--error contact-ajax-error';
    errorDiv.setAttribute('role', 'contentinfo');
    errorDiv.setAttribute('aria-label', Drupal.t('Error message'));

    const messagesContainer = document.createElement('div');
    messagesContainer.className = 'messages__container error-content';
    messagesContainer.setAttribute('role', 'alert');

    const messagesHeader = document.createElement('div');
    messagesHeader.className = 'error-header';

    const messagesContent = document.createElement('div');
    messagesContent.className = 'messages__content';
    messagesContent.textContent = message;

    const title = document.createElement('h2');
    title.className = 'visually-hidden';
    title.textContent = Drupal.t('Error message');
    messagesHeader.appendChild(title);

    const errorIcon = document.createElement('span');
    errorIcon.className = 'error-icon';
    errorIcon.setAttribute('aria-hidden', 'true');
    errorIcon.textContent = '⚠️';
    messagesHeader.appendChild(errorIcon);

    messagesContainer.appendChild(messagesHeader);
    messagesContainer.appendChild(messagesContent);
    errorDiv.appendChild(messagesContainer);

    element.innerHTML = '';
    element.appendChild(errorDiv);
    element.classList.add('has-error');
  }

})(jQuery, Drupal, once);
