<?php

namespace Drupal\contact_block_ajax\Controller;

use Drupal\Component\Utility\Html;
use Drupal\contact\Access\ContactPageAccess;
use Drupal\contact\ContactFormInterface;
use Drupal\contact\Entity\Message;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Flood\FloodInterface;
use Drupal\Core\Form\FormAjaxException;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\UserInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException;

/**
 * Returns responses for Contact Block AJAX routes.
 *
 * This controller handles the lazy loading of contact forms via AJAX,
 * supporting both site-wide and personal contact forms with proper
 * access control and rate limiting.
 */
class ContactBlockAjaxController extends ControllerBase {

  /**
   * The flood event name for form loading.
   */
  const FLOOD_EVENT = 'contact_block_ajax.form_load';

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The entity form builder.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected $entityFormBuilder;

  /**
   * The access check of personal contact.
   *
   * @var \Drupal\contact\Access\ContactPageAccess
   */
  protected $checkContactPageAccess;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The flood control mechanism.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * Constructs a ContactBlockAjaxController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   * @param \Drupal\contact\Access\ContactPageAccess $check_contact_page_access
   *   Check the access of personal contact.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood control mechanism.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $entity_display_repository,
    EntityFormBuilderInterface $entity_form_builder,
    ContactPageAccess $check_contact_page_access,
    AccountInterface $current_user,
    LoggerInterface $logger,
    FloodInterface $flood,
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityDisplayRepository = $entity_display_repository;
    $this->entityFormBuilder = $entity_form_builder;
    $this->checkContactPageAccess = $check_contact_page_access;
    $this->currentUser = $current_user;
    $this->logger = $logger;
    $this->flood = $flood;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('entity.form_builder'),
      $container->get('access_check.contact_personal'),
      $container->get('current_user'),
      $container->get('logger.factory')->get('contact_block_ajax'),
      $container->get('flood')
    );
  }

  /**
   * Loads and returns a contact form via AJAX.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Drupal\contact\ContactFormInterface|null $contact_form
   *   The contact form entity.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response containing the form.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Thrown when request validation fails.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Thrown when access is denied.
   * @throws \Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException
   *   Thrown when rate limit is exceeded.
   * @throws \Drupal\Core\Form\FormAjaxException
   *   Re-thrown to maintain normal form AJAX flow.
   */
  public function loadForm(Request $request, ?ContactFormInterface $contact_form = NULL): AjaxResponse {
    try {
      // Validate the request.
      $this->validateRequest($request);

      // Apply rate limiting.
      $this->checkRateLimit($request);

      // Validate parameters.
      $params = $this->validateParameters($request, $contact_form);

      // Create and configure the contact message, check access permissions.
      $contact_message = $this->createContactMessage($contact_form, $request);

      // Build the form.
      $form = $this->buildContactForm($contact_message, $params);

      // Create and return the AJAX response.
      return $this->createAjaxResponse($form, $params);
    }
    catch (FormAjaxException $e) {
      // FormAjaxException is part of Drupal's normal AJAX form flow.
      // We must re-throw it to allow the form submission/validation to work.
      throw $e;
    }
    catch (TooManyRequestsHttpException $e) {
      // Rate limiting exceptions can be thrown as-is.
      throw $e;
    }
    catch (\Exception $e) {
      // Log unexpected errors.
      $this->logError($e, $request, $contact_form);

      // Throw generic error for security.
      throw new BadRequestHttpException(
        'Unable to load the contact form. Please try again later.'
      );
    }
  }

  /**
   * Validates the incoming request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Thrown when request validation fails.
   */
  protected function validateRequest(Request $request): void {
    // Ensure this is an AJAX request.
    if (!$request->isXmlHttpRequest()) {
      throw new BadRequestHttpException('This endpoint only accepts AJAX requests.');
    }
  }

  /**
   * Validates and sanitizes request parameters.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Drupal\contact\ContactFormInterface|null $contact_form
   *   The contact form entity.
   *
   * @return array
   *   An array containing validated parameters.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Thrown when parameter validation fails.
   */
  protected function validateParameters(Request $request, ?ContactFormInterface $contact_form = NULL): array {
    // Validate contact form exists.
    if (!$contact_form) {
      throw new BadRequestHttpException('Contact form not found.');
    }

    // Validate and sanitize wrapper_id.
    $wrapper_id = $request->query->get('wrapper_id');
    if (empty($wrapper_id) || !is_string($wrapper_id)) {
      throw new BadRequestHttpException('Missing or invalid wrapper_id parameter.');
    }
    $wrapper_id = Html::getId($wrapper_id);

    // Validate and sanitize form display.
    $form_display = $request->query->get('display', 'default');
    if (!is_string($form_display)) {
      throw new BadRequestHttpException('Invalid display parameter.');
    }

    // Verify form display exists.
    $available_displays = $this->entityDisplayRepository
      ->getFormModeOptions('contact_message');
    if (!isset($available_displays[$form_display])) {
      throw new BadRequestHttpException('Invalid form display mode.');
    }

    return [
      'wrapper_id' => $wrapper_id,
      'form_display' => $form_display,
      'block_selector' => '#' . $wrapper_id . '.ajax-contact-form-container',
      'form_wrapper_id' => $wrapper_id . '-form',
    ];
  }

  /**
   * Creates a contact message entity and checks access permissions.
   *
   * @param \Drupal\contact\ContactFormInterface $contact_form
   *   The contact form entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Drupal\contact\Entity\Message
   *   The created contact message entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Thrown when message creation fails.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Thrown when access is denied.
   */
  protected function createContactMessage(ContactFormInterface $contact_form, Request $request): Message {
    /**
     * @var \Drupal\contact\Entity\Message $contact_message
     */
    $contact_message = $this->entityTypeManager
      ->getStorage('contact_message')
      ->create(
        [
          'contact_form' => $contact_form->id(),
        ]
      );

    // Handle personal contact forms.
    if ($contact_message->isPersonal()) {
      $uid = $request->query->get('user');

      if (empty($uid) || !is_numeric($uid)) {
        throw new BadRequestHttpException('Invalid user parameter for personal contact form.');
      }

      /**
       * @var \Drupal\user\UserInterface|null $user
       */
      $user = $this->entityTypeManager
        ->getStorage('user')
        ->load($uid);

      if (!$user instanceof UserInterface) {
        throw new BadRequestHttpException('User not found.');
      }

      // Check personal contact access.
      $access = $this->checkContactPageAccess->access($user, $this->currentUser);
      if (!$access->isAllowed()) {
        throw new AccessDeniedHttpException('Access denied to personal contact form.');
      }

      $contact_message->set('recipient', $user);
    }
    else {
      // Check site-wide contact form access.
      $access = $contact_form->access('view', $this->currentUser, TRUE);
      if (!$access->isAllowed()) {
        throw new AccessDeniedHttpException('Access denied to contact form.');
      }
    }

    return $contact_message;
  }

  /**
   * Builds the contact form.
   *
   * @param \Drupal\contact\Entity\Message $contact_message
   *   The contact message entity.
   * @param array $params
   *   The validated parameters.
   *
   * @return array
   *   The form render array.
   */
  protected function buildContactForm(Message $contact_message, array $params): array {
    // Build the form with custom parameters.
    $form = $this->entityFormBuilder->getForm(
      $contact_message,
      $params['form_display'],
      [
        'contact_block_ajax_form' => TRUE,
        'wrapper_id' => $params['form_wrapper_id'],
      ]
    );

    // Store the form display for reference.
    $form['#form_display'] = $params['form_display'];

    // Wrap form in container for proper styling and targeting.
    return [
      'contact_wrapper' => [
        '#type' => 'container',
        '#attributes' => [
          'id' => $params['form_wrapper_id'],
          'class' => [Html::getClass($params['form_wrapper_id'])],
        ],
        'form' => $form,
      ],
    ];
  }

  /**
   * Creates the AJAX response.
   *
   * @param array $form
   *   The form render array.
   * @param array $params
   *   The validated parameters.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  protected function createAjaxResponse(array $form, array $params): AjaxResponse {
    $response = new AjaxResponse();

    // Add commands to update the DOM.
    $response->addCommand(new HtmlCommand($params['block_selector'], $form));
    $response->addCommand(new InvokeCommand($params['block_selector'], 'removeClass', ['is-loading']));
    $response->addCommand(new InvokeCommand($params['block_selector'], 'addClass', ['form-loaded']));
    $response->addCommand(new InvokeCommand($params['block_selector'], 'attr', [
      'aria-busy',
      'false',
    ]));

    // Set cache headers to prevent caching of dynamic forms.
    $response->setMaxAge(0);
    $response->headers->set('X-Robots-Tag', 'noindex');

    return $response;
  }

  /**
   * Checks if the request exceeds the rate limit.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException
   *   Thrown when rate limit is exceeded.
   */
  protected function checkRateLimit(Request $request): void {
    // Load rate limit configuration.
    $rate_limit_config = $this->configFactory
      ->get('contact_block_ajax.form_load_rate_limit');

    // Skip if rate limiting is disabled.
    if (!$rate_limit_config->get('enabled')) {
      return;
    }

    // Get rate limit parameters.
    $threshold = (int) $rate_limit_config->get('limit');
    $window = (int) $rate_limit_config->get('interval');
    $identifier = $request->getClientIp();

    // Check if request exceeds the allowed rate.
    if (!$this->flood->isAllowed(self::FLOOD_EVENT, $threshold, $window, $identifier)) {
      // Log the rate limit violation.
      $this->logger->warning(
        'Rate limit exceeded for contact form loading. IP: @ip, Limit: @limit/@window seconds',
        [
          '@ip' => $identifier,
          '@limit' => $threshold,
          '@window' => $window,
        ]
      );

      // Throw exception with retry-after header.
      throw new TooManyRequestsHttpException(
        $window,
        sprintf('Too many form load requests. Please wait %d seconds before trying again.', $window),
      );
    }

    // Register this request in the flood table.
    $this->flood->register(self::FLOOD_EVENT, $window, $identifier);
  }

  /**
   * Logs errors for administrative review.
   *
   * @param \Exception $exception
   *   The exception that was caught.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Drupal\contact\ContactFormInterface|null $contact_form
   *   The contact form entity.
   */
  protected function logError(\Exception $exception, Request $request, ?ContactFormInterface $contact_form = NULL): void {
    // Build context array for logging.
    $context = [
      '@message' => $exception->getMessage(),
      '@code' => $exception->getCode(),
      '@ip' => $request->getClientIp(),
      '@path' => $request->getPathInfo(),
      '@form' => $contact_form ? $contact_form->id() : 'unknown',
      '@user' => $this->currentUser->id(),
    ];

    // Add request parameters for debugging.
    $context['@params'] = json_encode(
      [
        'wrapper_id' => $request->query->get('wrapper_id'),
        'display' => $request->query->get('display'),
        'user' => $request->query->get('user'),
      ]
    );

    // Log based on exception type.
    if ($exception instanceof AccessDeniedHttpException) {
      $this->logger->warning(
        'Access denied loading contact form @form for user @user from IP @ip. Error: @message',
        $context
      );
    }
    elseif ($exception instanceof BadRequestHttpException) {
      $this->logger->notice(
        'Bad request loading contact form @form from IP @ip. Error: @message. Parameters: @params',
        $context
      );
    }
    else {
      $this->logger->error(
        'Unexpected error loading contact form @form. Error: @message. Code: @code. Path: @path',
        $context
      );
    }
  }

}
