<?php

namespace Drupal\contact_block_ajax\Plugin\Block;

use Drupal\contact\Entity\Message;
use Drupal\contact\Entity\ContactForm;
use Drupal\Component\Utility\Html;
use Drupal\contact\Access\ContactPageAccess;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a block for Contact Block AJAX.
 *
 * This block lazy-loads Drupal contact forms via AJAX using Intersection
 * Observer API, improving page load performance by deferring form rendering
 * until the block enters the viewport.
 *
 * Features:
 * - Lazy loading with Intersection Observer
 * - AJAX form submission
 * - Multiple contact form support
 * - Custom form display mode support
 * - Integration with CAPTCHA, reCAPTCHA, and Honeypot
 *
 * @Block(
 *   id = "contact_block_ajax",
 *   admin_label = @Translation("Contact Block AJAX"),
 *   category = @Translation("Contact Block"),
 * )
 */
class ContactBlockAjax extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Contact block AJAX prefix.
   *
   * @var string
   */
  public $contactBlockAjaxPrefix = 'contact_block_ajax_';

  /**
   * The contact form configuration entity.
   *
   * @var \Drupal\contact\Entity\ContactForm
   */
  protected $contactForm;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $routeMatch;

  /**
   * The access check of personal contact.
   *
   * @var \Drupal\contact\Access\ContactPageAccess
   */
  protected $checkContactPageAccess;

  /**
   * Constructs a ContactBlockAjax object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $route_match
   *   The route match service.
   * @param \Drupal\contact\Access\ContactPageAccess $check_contact_page_access
   *   Check the access of personal contact.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $entity_display_repository,
    CurrentRouteMatch $route_match,
    ContactPageAccess $check_contact_page_access,
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityDisplayRepository = $entity_display_repository;
    $this->routeMatch = $route_match;
    $this->checkContactPageAccess = $check_contact_page_access;
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
          $configuration,
          $plugin_id,
          $plugin_definition,
          $container->get('config.factory'),
          $container->get('entity_type.manager'),
          $container->get('entity_display.repository'),
          $container->get('current_route_match'),
          $container->get('access_check.contact_personal')
      );
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    $contact_form = $this->getContactForm();
    $contact_message = $this->createContactMessage();

    // Deny access when the configured contact form has been deleted.
    if (empty($contact_form)) {
      return AccessResult::forbidden();
    }

    if ($contact_message->isPersonal()) {
      /**
       * @var \Drupal\user\Entity\User $user
       */
      $user = $this->routeMatch->getParameter('user');

      if (empty($user)) {
        return AccessResult::forbidden();
      }

      return $this->checkContactPageAccess->access($user, $account);
    }

    return $contact_form->access('view', $account, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $default_form = $this->configFactory
      ->get('contact.settings')
      ->get('default_form');

    return [
      'label' => $this->t('Contact Block AJAX'),
      'contact_form' => $default_form ?? '',
      'form_display' => 'default',
      'wrapper_id' => $this->contactBlockAjaxPrefix . 'wrapper',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $options = $this->entityTypeManager
      ->getStorage('contact_form')
      ->loadMultiple();

    // Build options array.
    $contact_form_options = [];
    foreach ($options as $key => $option) {
      $contact_form_options[$key] = $option->label();
    }

    $form['contact_form'] = [
      '#type' => 'select',
      '#title' => $this->t('Contact form'),
      '#description' => $this->t('Select the contact form to display. Personal contact forms are not supported.'),
      '#options' => $contact_form_options,
      '#default_value' => $this->configuration['contact_form'],
      '#required' => TRUE,
    ];

    $form['form_display'] = [
      '#type' => 'select',
      '#title' => $this->t('Form display'),
      '#description' => $this->t('Select the form display mode to use.'),
      '#options' => $this->entityDisplayRepository
        ->getFormModeOptions('contact_message'),
      '#default_value' => $this->configuration['form_display'],
      '#required' => TRUE,
    ];

    $form['wrapper_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Wrapper ID'),
      '#description' => $this->t('Unique wrapper ID for this block instance.'),
      '#default_value' => $this->configuration['wrapper_id'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['contact_form'] = $form_state->getValue('contact_form');
    $this->configuration['form_display'] = $form_state->getValue('form_display');
    $this->configuration['wrapper_id'] = $form_state->getValue('wrapper_id');
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state) {
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $form_state->getValue('wrapper_id'))) {
      $form_state->setErrorByName(
            'wrapper_id',
            $this->t('Wrapper ID must contain only letters, numbers, and underscores.')
        );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [];

    /**
* @var \Drupal\contact\Entity\ContactForm $contact_form
*/
    $contact_form = $this->getContactForm();

    if ($contact_form) {
      $contact_message = $this->createContactMessage();
      $form_id = $contact_form->id();
      $form_display = $this->configuration['form_display'];
      $wrapper_id = Html::getUniqueId($this->contactBlockAjaxPrefix . $this->configuration['wrapper_id']);
      $options = [
        'query' => [
          'display' => $form_display,
          'wrapper_id' => $wrapper_id,
        ],
      ];

      if ($contact_message->isPersonal()) {
        $user = $this->routeMatch->getParameter('user');
        if ($user) {
          $options['query']['user'] = $user->id();
        }
      }

      // Build AJAX URL.
      $ajax_url = Url::fromRoute(
            'contact_block_ajax.load_form', [
              'contact_form' => $form_id,
            ], $options
        )->toString();

      $build = [
        '#theme' => 'contact_block_ajax',
        '#wrapper_id' => $wrapper_id,
        '#form_id' => $form_id,
        '#form_display' => $form_display,
        '#ajax_url' => $ajax_url,
        '#attached' => [
          'library' => [
            'contact_block_ajax/contact_block_ajax',
          ],
          'drupalSettings' => [
            'contactBlockAjax' => [
              'wrapperSelector' => '#' . $wrapper_id,
              'threshold' => 0.1,
              'rootMargin' => '50px 0px',
            ],
          ],
        ],
        '#cache' => [
          'contexts' => ['user.permissions'],
        ],
      ];

      $build['#contextual_links']['contact_block'] = [
        'route_parameters' => ['contact_form' => $contact_form->id()],
      ];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = array_merge_recursive(
          parent::calculateDependencies(), [
            'config' => [],
          ]
      );

    if ($contact_form = $this->getContactForm()) {
      $dependencies['config'][] = $contact_form->getConfigDependencyName();
    }

    return $dependencies;
  }

  /**
   * Loads the contact form entity.
   *
   * @return \Drupal\contact\Entity\ContactForm|null
   *   The contact form configuration entity. NULL if the entity does not exist.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getContactForm(): ?ContactForm {
    if (!isset($this->contactForm)) {
      if (isset($this->configuration['contact_form'])) {
        $this->contactForm = $this->entityTypeManager
          ->getStorage('contact_form')
          ->load($this->configuration['contact_form']);
      }
    }
    return $this->contactForm;
  }

  /**
   * Creates the contact message entity without saving it.
   *
   * @return \Drupal\contact\Entity\Message|null
   *   The contact message entity. NULL if the entity does not exist.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function createContactMessage(): ?Message {
    $contact_message = NULL;
    $contact_form = $this->getContactForm();
    if ($contact_form) {
      $contact_message = $this->entityTypeManager
        ->getStorage('contact_message')
        ->create(['contact_form' => $contact_form->id()]);
    }
    return $contact_message;
  }

}
