<?php

namespace Drupal\Tests\contact_block_ajax\Functional;

use Drupal\contact\ContactFormInterface;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\contact\Entity\ContactForm;

/**
 * Tests Contact Block AJAX functionality.
 *
 * @group contact_block_ajax
 */
final class ContactBlockAjaxTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'contact',
    'block',
    'contact_block_ajax',
  ];

  /**
   * Test contact form.
   *
   * @var \Drupal\contact\ContactFormInterface
   */
  protected ContactFormInterface $contactForm;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a contact form.
    $this->contactForm = ContactForm::create([
      'id' => 'test_form',
      'label' => 'Test form',
      'recipients' => ['admin@example.com'],
    ]);
    $this->contactForm->save();
  }

  /**
   * Tests that the block can be placed.
   */
  public function testBlockPlacement() {
    $admin_user = $this->drupalCreateUser([
      'administer blocks',
      'access site-wide contact form',
    ]);
    $this->drupalLogin($admin_user);

    $this->drupalGet('admin/structure/block/add/contact_block_ajax/stark');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldExists('settings[contact_form]');
    $this->assertSession()->fieldExists('settings[form_display]');
    $this->assertSession()->fieldExists('settings[wrapper_id]');
  }

  /**
   * Tests AJAX form loading endpoint with permission.
   */
  public function testAjaxEndpointWithPermission() {
    // Create user with permission.
    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    // Build the URL.
    $url = $this->buildAjaxUrl($this->contactForm->id());

    // Make AJAX request.
    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    $this->assertSession()->statusCodeEquals(200);

    // Verify it returns JSON response.
    $this->assertSession()->responseHeaderContains('Content-Type', 'application/json');
  }

  /**
   * Tests AJAX endpoint without permission.
   */
  public function testAjaxEndpointWithoutPermission() {
    // User without permission.
    $user = $this->drupalCreateUser([]);
    $this->drupalLogin($user);

    // Build the URL.
    $url = $this->buildAjaxUrl($this->contactForm->id());

    // Make AJAX request.
    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Should be denied, bad request with generic error is shown.
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Tests AJAX endpoint requires AJAX request.
   */
  public function testAjaxEndpointRequiresAjax() {
    // Create user with permission.
    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    // Build the URL.
    $url = $this->buildAjaxUrl($this->contactForm->id());

    // Make regular (non-AJAX) request - should fail.
    $this->drupalGet($url);

    // Should return 400 Bad Request.
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Tests block rendering.
   */
  public function testBlockRendering() {
    // Place the block.
    $this->drupalPlaceBlock('contact_block_ajax', [
      'id' => 'contact_ajax_test',
      'contact_form' => $this->contactForm->id(),
      'form_display' => 'default',
    ]);

    // Create user with permission.
    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    // Visit page with block.
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // Check that container exists.
    $this->assertSession()->elementExists('css', '.ajax-contact-form-container');

    // Check that the wrapper has the correct class.
    $this->assertSession()->elementExists('css', '.contact-block-ajax-wrapper');

    // Verify JavaScript file is attached (check for the actual filename).
    $this->assertSession()->responseContains('contact-block-ajax.js');

    // Verify CSS file is attached.
    $this->assertSession()->responseContains('contact-block-ajax.css');

    // Check data attributes.
    $this->assertSession()->elementAttributeExists('css', '.ajax-contact-form-container', 'data-ajax-url');
    $this->assertSession()->elementAttributeContains('css', '.ajax-contact-form-container', 'data-loaded', 'false');

    // Check accessibility attributes.
    $this->assertSession()->elementAttributeExists('css', '.ajax-contact-form-container', 'aria-live');
    $this->assertSession()->elementAttributeExists('css', '.ajax-contact-form-container', 'aria-busy');

    // Verify the AJAX URL is properly formed.
    $container = $this->getSession()->getPage()->find('css', '.ajax-contact-form-container');
    $ajax_url = $container->getAttribute('data-ajax-url');
    $this->assertStringContainsString('/contact-block-ajax/' . $this->contactForm->id(), $ajax_url);
    $this->assertStringContainsString('wrapper_id=', $ajax_url);
    $this->assertStringContainsString('display=default', $ajax_url);
  }

  /**
   * Tests that invalid wrapper_id is rejected.
   */
  public function testInvalidWrapperId() {
    // Create user with permission.
    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    // Build URL without wrapper_id.
    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => $this->contactForm->id(),
    ], [
      'query' => [
        'display' => 'default',
      ],
    ]);

    // Make AJAX request without wrapper_id.
    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Should return 400 Bad Request.
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Tests that invalid form display is rejected.
   */
  public function testInvalidFormDisplay() {
    // Create user with permission.
    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    // Build URL with invalid display.
    $wrapper_id = 'test-wrapper-123';
    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => $this->contactForm->id(),
    ], [
      'query' => [
        'display' => 'invalid_display_mode',
        'wrapper_id' => $wrapper_id,
      ],
    ]);

    // Make AJAX request with invalid display.
    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Should return 400 Bad Request.
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Tests personal contact form functionality.
   *
   * Tests the complete functionality of personal contact forms including:
   * - Valid user access
   * - Permission checking
   * - User enumeration prevention
   * - Error handling with generic messages.
   */
  public function testPersonalContactForm() {
    // Setup: Enable personal contact forms globally.
    $this->config('contact.settings')
      ->set('user_default_enabled', TRUE)
      ->save();

    $recipient_enabled = $this->drupalCreateUser([
      'access user contact forms',
    ], 'recipient_enabled');

    $sender_authorized = $this->drupalCreateUser([
      'access site-wide contact form',
      'access user contact forms',
    ], 'sender_authorized');

    $sender_unauthorized = $this->drupalCreateUser([
      'access content',
    ], 'sender_unauthorized');

    // Place the contact block on user pages.
    $this->drupalPlaceBlock('contact_block_ajax', [
      'id' => 'contact_ajax_personal_test',
      'contact_form' => 'personal',
      'form_display' => 'default',
      'wrapper_id' => 'personal-contact-wrapper',
      'visibility' => [
        'request_path' => [
          'pages' => '/user/*',
        ],
      ],
      'region' => 'content',
    ]);

    $this->drupalLogin($sender_authorized);
    $this->drupalGet('user/' . $recipient_enabled->id() . '/contact');
    $this->assertSession()->statusCodeEquals(200);

    // Verify block elements exist.
    $this->assertSession()->elementExists('css', '.contact-block-ajax-wrapper');
    $container = $this->assertSession()->elementExists('css', '.ajax-contact-form-container');

    // Verify AJAX URL contains correct user parameter.
    $ajax_url = $container->getAttribute('data-ajax-url');
    $this->assertNotEmpty($ajax_url, 'AJAX URL should be present');
    $this->assertStringContainsString('user=' . $recipient_enabled->id(), $ajax_url);
    $this->assertStringContainsString('/contact-block-ajax/personal', $ajax_url);

    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => 'personal',
    ], [
      'query' => [
        'display' => 'default',
        'wrapper_id' => 'test-personal-wrapper',
        'user' => $recipient_enabled->id(),
      ],
    ]);

    $response = $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseHeaderContains('Content-Type', 'application/json');

    // Verify response contains AJAX commands.
    $json = json_decode($response, TRUE);
    $this->assertIsArray($json, 'Response should be valid JSON array');
    $this->assertNotEmpty($json, 'Response should contain AJAX commands');

    // Non-existent User (User Enumeration Prevention)
    $non_existent_user_id = 999999;

    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => 'personal',
    ], [
      'query' => [
        'display' => 'default',
        'wrapper_id' => 'test-personal-wrapper',
        'user' => $non_existent_user_id,
      ],
    ]);

    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Should return generic 400 error (not 404) to prevent user enumeration.
    $this->assertSession()->statusCodeEquals(400);

    // Verify generic error message (should not reveal user doesn't exist).
    $response_text = $this->getSession()->getPage()->getText();
    $this->assertStringNotContainsString('user not found', strtolower($response_text));
    $this->assertStringNotContainsString('does not exist', strtolower($response_text));

    // Access Without Permission.
    $this->drupalLogin($sender_unauthorized);

    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => 'personal',
    ], [
      'query' => [
        'display' => 'default',
        'wrapper_id' => 'test-personal-wrapper',
        'user' => $recipient_enabled->id(),
      ],
    ]);

    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Should return generic 400 error.
    $this->assertSession()->statusCodeEquals(400);

    // Missing User Parameter.
    $this->drupalLogin($sender_authorized);

    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => 'personal',
    ], [
      'query' => [
        'display' => 'default',
        'wrapper_id' => 'test-personal-wrapper',
        // 'user' parameter intentionally missing.
      ],
    ]);

    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Should return 400 Bad Request.
    $this->assertSession()->statusCodeEquals(400);

    // Anonymous User Access.
    $this->drupalLogout();

    $url = Url::fromRoute('contact_block_ajax.load_form', [
      'contact_form' => 'personal',
    ], [
      'query' => [
        'display' => 'default',
        'wrapper_id' => 'test-personal-wrapper',
        'user' => $recipient_enabled->id(),
      ],
    ]);

    $this->drupalGet($url, [], [
      'X-Requested-With' => 'XMLHttpRequest',
    ]);

    // Anonymous users should be denied (400).
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Builds an AJAX URL for testing.
   *
   * @return \Drupal\Core\Url
   *   The URL object.
   */
  protected function buildAjaxUrl(string $contact_form): Url {
    return Url::fromRoute(
      'contact_block_ajax.load_form',
      ['contact_form' => $contact_form],
      [
        'query' => [
          'display' => 'default',
          'wrapper_id' => 'test-wrapper-' . $this->randomMachineName(),
        ],
      ]
    );
  }

}
