<?php

namespace Drupal\Tests\contact_block_ajax\Functional;

use Drupal\contact\ContactFormInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\contact\Entity\ContactForm;

/**
 * Tests Contact Block AJAX rate limiting functionality.
 *
 * @group contact_block_ajax
 */
final class RateLimitTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'contact',
    'block',
    'contact_block_ajax',
  ];

  /**
   * Test contact form.
   *
   * @var \Drupal\contact\ContactFormInterface
   */
  protected ContactFormInterface $contactForm;

  /**
   * Configuration object for rate limiting.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $rateLimitConfig;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a contact form.
    $this->contactForm = ContactForm::create([
      'id' => 'test_form',
      'label' => 'Test form',
      'recipients' => ['admin@example.com'],
    ]);
    $this->contactForm->save();

    // Get rate limit configuration.
    $this->rateLimitConfig = $this
      ->config('contact_block_ajax.form_load_rate_limit');
  }

  /**
   * Tests rate limit configuration form.
   */
  public function testRateLimitConfigurationForm(): void {
    $admin_user = $this->drupalCreateUser(['administer form load rate limit']);
    $this->drupalLogin($admin_user);

    // Visit rate limit configuration page.
    $this->drupalGet('admin/config/people/contact-block-ajax-rate-limit');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Form Load Rate Limiting');

    // Check form elements exist.
    $this->assertSession()->fieldExists('enabled');
    $this->assertSession()->fieldExists('limit');
    $this->assertSession()->fieldExists('interval');

    // Enable rate limiting with custom values.
    $this->submitForm([
      'enabled' => TRUE,
      'limit' => '5',
      'interval' => '60',
    ], 'Save configuration');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify configuration was saved.
    $config = $this->config('contact_block_ajax.form_load_rate_limit');
    $this->assertTrue($config->get('enabled'));
    $this->assertEquals(5, $config->get('limit'));
    $this->assertEquals(60, $config->get('interval'));
  }

  /**
   * Tests rate limit configuration form without permission.
   */
  public function testRateLimitConfigurationFormWithoutPermission(): void {
    $user = $this->drupalCreateUser([]);
    $this->drupalLogin($user);

    // Try to access rate limit configuration page.
    $this->drupalGet('admin/config/people/contact-block-ajax-rate-limit');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that rate limiting is disabled by default.
   */
  public function testRateLimitDisabledByDefault(): void {
    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    // Verify rate limiting is disabled by default.
    $this->assertFalse($this->rateLimitConfig->get('enabled'));

    // Build the AJAX URL.
    $url = $this->buildAjaxUrl();

    // Should be able to make many requests without limit.
    for ($i = 0; $i < 15; $i++) {
      $this->drupalGet($url, [], ['X-Requested-With' => 'XMLHttpRequest']);
      $this->assertSession()->statusCodeEquals(200);
    }
  }

  /**
   * Tests that rate limiting blocks excessive requests.
   */
  public function testRateLimitBlocksExcessiveRequests(): void {
    // Enable rate limiting with strict settings.
    $this->rateLimitConfig
      ->set('enabled', TRUE)
      ->set('limit', 3)
      ->set('interval', 60)
      ->save();

    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    $url = $this->buildAjaxUrl();

    // First 3 requests should succeed.
    for ($i = 0; $i < 3; $i++) {
      $this->drupalGet($url, [], ['X-Requested-With' => 'XMLHttpRequest']);
      $this->assertSession()->statusCodeEquals(200);
    }

    // 4th request should be blocked with 429 status.
    $this->drupalGet($url, [], ['X-Requested-With' => 'XMLHttpRequest']);
    $this->assertSession()->statusCodeEquals(429);
    $this->assertSession()->responseContains('Too many');
  }

  /**
   * Tests that rate limit error includes retry-after information.
   */
  public function testRateLimitRetryAfterHeader(): void {
    // Enable rate limiting.
    $this->rateLimitConfig
      ->set('enabled', TRUE)
      ->set('limit', 2)
      ->set('interval', 120)
      ->save();

    $user = $this->drupalCreateUser(['access site-wide contact form']);
    $this->drupalLogin($user);

    $url = $this->buildAjaxUrl();

    // Make requests until rate limited.
    for ($i = 0; $i < 2; $i++) {
      $this->drupalGet($url, [], ['X-Requested-With' => 'XMLHttpRequest']);
    }

    // Next request should be rate limited.
    $this->drupalGet($url, [], ['X-Requested-With' => 'XMLHttpRequest']);
    $this->assertSession()->statusCodeEquals(429);

    // Check for Retry-After header (not directly testable in BrowserTestBase).
    // Instead, verify the error message mentions the time window.
    $content = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('wait', strtolower($content));
  }

  /**
   * Builds an AJAX URL for testing.
   *
   * @return \Drupal\Core\Url
   *   The URL object.
   */
  protected function buildAjaxUrl(): Url {
    return Url::fromRoute(
      'contact_block_ajax.load_form',
      ['contact_form' => $this->contactForm->id()],
      [
        'query' => [
          'display' => 'default',
          'wrapper_id' => 'test-wrapper-' . $this->randomMachineName(),
        ],
      ]
    );
  }

}
