# Contact Slack Integration

This module integrates Drupal's core Contact module with Slack,
sending notifications to specified Slack channels when contact forms
are submitted.

## Features

- **Real-time Notifications**: Instant Slack alerts for new contact
  form submissions
- **Multiple Channel Support**: Different channels for different contact forms
- **Configurable Fields**: Choose which fields to include in notifications
- **Per-Form Configuration**: Individual Slack settings for each contact form
- **Custom Message Text**: Optional default message to prepend to notifications
- **Test Functionality**: Send test messages to verify Slack integration
- **Error Handling**: Comprehensive retry logic and error logging
- **Admin Integration**: Seamless integration with Drupal's contact
  form admin interface

## Requirements

- Drupal 10.0+ or Drupal 11.0+
- PHP 8.1 or higher
- Contact module (Drupal core)
- A Slack workspace with webhook permissions

## Installation

1. **Install via Composer** (recommended):
   ```bash
   composer require drupal/contact_slack
   ```

2. **Or download manually**:
   - Download the module to `modules/contrib/contact_slack`

3. **Enable the module**:
   ```bash
   drush en contact_slack
   ```

## Configuration

### 1. Set up Slack Webhook

1. Go to your Slack workspace
2. Navigate to **Apps** → **Incoming Webhooks**
3. Click **Add to Slack**
4. Choose a channel and click **Add Incoming WebHooks integration**
5. Copy the **Webhook URL** (starts with `https://hooks.slack.com/services/`)

### 2. Configure Global Settings

1. Navigate to **Configuration** → **Contact Slack settings**
2. Enter your **Slack Webhook URL**
3. Configure retry settings (optional)
4. Click **Send Test Message** to verify the integration

### 3. Configure Per-Form Settings

1. Go to **Structure** → **Contact forms**
2. Click **Edit** on any contact form
3. Click the **Slack settings** operation link
4. Configure form-specific settings:
   - Enable/disable notifications for this form
   - Add custom webhook URLs (optional, uses global default if empty)
   - Select which fields to include
   - Enable system information (IP, user agent)
   - Add default message text to prepend to notifications (optional)

## Usage

Once configured, the module automatically:

1. **Captures Contact Submissions**: Uses `hook_contact_message_insert()`
   to detect new submissions
2. **Formats Messages**: Creates rich Slack messages with contact details
3. **Sends Notifications**: Posts to configured Slack channels
4. **Handles Errors**: Retries failed sends and logs errors
5. **Provides Feedback**: Logs successful sends for monitoring

## Message Format

Default Slack messages include:

```
🔔 New Contact Form Submission

📧 From: John Doe <john@example.com>
📝 Subject: General Inquiry
🏢 Form: Website Contact Form
⏰ Submitted: January 26, 2025 - 13:45

Message:
Hello, I have a question about your services...
```

**With optional default message text:**

```
⚠️ URGENT: Customer Support Request

🔔 New Contact Form Submission

📧 From: John Doe <john@example.com>
📝 Subject: General Inquiry
...
```

## Permissions

The module adds one permission:

- **Administer Contact Slack integration**: Configure Slack settings
  for contact forms

Users need both this permission AND **Administer contact forms** to
access Slack settings.

## Troubleshooting

### Test Message Fails

1. **Check Webhook URL**: Ensure it starts with
   `https://hooks.slack.com/services/`
2. **Verify Slack App**: Make sure the Slack app is still installed
   in your workspace
3. **Check Permissions**: Ensure the webhook has permission to post
   to the specified channel
4. **Review Logs**: Check **Reports** → **Recent log messages** for
   detailed errors

### Contact Form Notifications Not Sent

1. **Check Form Settings**: Ensure Slack notifications are enabled for
   the specific form
2. **Verify Webhook URL**: Make sure the webhook URL is correctly
   configured
3. **Check Channel**: Ensure the channel exists and the webhook can access it
4. **Review Logs**: Look for error messages in the system log

### Common Issues

- **Invalid Channel Name**: Channel names must start with `#`
  (channels) or `@` (direct messages)
- **Webhook Expired**: Slack webhooks can expire; regenerate if needed
- **Network Issues**: Check firewall settings if requests are blocked
- **Rate Limiting**: Slack may rate limit; the module includes retry logic

## API

### Services

Access the Slack service in custom code:

```php
$slack_service = \Drupal::service('contact_slack.notification');

// Send a test message
$slack_service->sendTestMessage('#test-channel');

// Send notification for a contact message
$slack_service->sendContactNotification($contact_message);
```

## Configuration Management

All settings are exportable via Drupal's configuration management:

- **Global settings**: `contact_slack.settings`
- **Per-form settings**: Stored in each contact form's `third_party_settings`

Export/import works normally with `drush config:export` and
`drush config:import`.

## Security

- **Webhook URLs** are stored securely in Drupal configuration
- **Input sanitization** prevents malicious content in Slack messages
- **Permission-based access** controls who can configure Slack settings
- **Optional system info** (IP addresses) can be disabled for privacy

## Support

- **Issue Queue**: Report bugs and request features on Drupal.org
- **Documentation**: Additional help available in the module's help pages
- **Community**: Get help from the Drupal community

## License

This project is licensed under the GPL-2.0+ license - the same as Drupal core.

## Contributing

Contributions are welcome! Please:

1. Fork the project
2. Create a feature branch
3. Make your changes
4. Add tests if applicable
5. Submit a pull request

## Changelog

### 1.0.0
- Initial release
- Basic Slack integration with contact forms
- Per-form configuration
- Test message functionality
- Comprehensive error handling
