<?php

namespace Drupal\contact_slack\Form;

use Drupal\contact\ContactFormInterface;
use Drupal\contact_slack\SlackWebhookValidationTrait;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Slack settings for individual contact forms.
 */
class ContactFormSlackForm extends FormBase {

  use SlackWebhookValidationTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'contact_form_slack_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\contact\ContactFormInterface|null $contact_form
   *   The contact form entity.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?ContactFormInterface $contact_form = NULL): array {
    if (!$contact_form) {
      return $form;
    }

    $form_state->set('contact_form', $contact_form);

    $form['slack_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Slack Integration Settings'),
      '#description' => $this->t('Configure Slack notifications for the %form contact form.', [
        '%form' => $contact_form->label(),
      ]),
      '#open' => TRUE,
    ];

    $form['slack_settings']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Slack notifications for this form'),
      '#description' => $this->t('Enable Slack notifications when this contact form is submitted.'),
      '#default_value' => $contact_form->getThirdPartySetting('contact_slack', 'enabled', FALSE),
    ];

    $webhook_urls = $contact_form->getThirdPartySetting('contact_slack', 'webhook_urls', []);
    $form['slack_settings']['webhook_urls'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Webhook URLs (Optional)'),
      '#description' => $this->t('Leave empty to use the default webhook URL. Enter custom Slack webhook URLs (one per line) to send this form to multiple channels.'),
      '#default_value' => !empty($webhook_urls) ? implode("\n", $webhook_urls) : '',
      '#rows' => 5,
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['slack_settings']['default_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Default Message Text (Optional)'),
      '#description' => $this->t('Optional text to prepend to all Slack notifications from this form. This can be used for context, instructions, or additional information.'),
      '#default_value' => $contact_form->getThirdPartySetting('contact_slack', 'default_message', ''),
      '#rows' => 3,
      '#placeholder' => $this->t('e.g., "New inquiry from company website" or "Urgent: Customer support request"'),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['slack_settings']['include_fields'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Include Fields'),
      '#description' => $this->t('Select which fields to include in Slack notifications.'),
      '#options' => [
        'name' => $this->t('Sender Name'),
        'mail' => $this->t('Sender Email'),
        'subject' => $this->t('Subject'),
        'message' => $this->t('Message'),
        'form_name' => $this->t('Form Name'),
      ],
      '#default_value' => $contact_form->getThirdPartySetting('contact_slack', 'include_fields', [
        'name', 'mail', 'subject', 'message', 'form_name',
      ]),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['slack_settings']['include_system_info'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include System Information'),
      '#description' => $this->t('Include additional system information like IP address and user agent in notifications.'),
      '#default_value' => $contact_form->getThirdPartySetting('contact_slack', 'include_system_info', FALSE),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Slack settings'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Validate webhook URLs if provided.
    $webhook_urls_text = $form_state->getValue('webhook_urls');
    if (!empty($webhook_urls_text)) {
      $webhook_urls = array_filter(array_map('trim', explode("\n", $webhook_urls_text)));
      foreach ($webhook_urls as $line_number => $webhook_url) {
        if (!$this->validateSlackWebhookUrl($webhook_url)) {
          $form_state->setErrorByName('webhook_urls', $this->t('Invalid Slack webhook URL on line @line: @url. Must start with https://hooks.slack.com/services/', [
            '@line' => $line_number + 1,
            '@url' => $webhook_url,
          ]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\contact\ContactFormInterface $contact_form */
    $contact_form = $form_state->get('contact_form');

    // Save third party settings.
    $contact_form->setThirdPartySetting('contact_slack', 'enabled', $form_state->getValue('enabled'));

    // Process webhook URLs.
    $webhook_urls_text = $form_state->getValue('webhook_urls');
    $webhook_urls = !empty($webhook_urls_text) ? array_filter(array_map('trim', explode("\n", $webhook_urls_text))) : [];
    $contact_form->setThirdPartySetting('contact_slack', 'webhook_urls', array_values($webhook_urls));

    $contact_form->setThirdPartySetting('contact_slack', 'include_fields', array_filter($form_state->getValue('include_fields')));
    $contact_form->setThirdPartySetting('contact_slack', 'include_system_info', $form_state->getValue('include_system_info'));
    $contact_form->setThirdPartySetting('contact_slack', 'default_message', trim($form_state->getValue('default_message')));

    $contact_form->save();

    $this->messenger()->addStatus($this->t('Slack settings for %form have been saved.', [
      '%form' => $contact_form->label(),
    ]));
  }

}
