<?php

namespace Drupal\contact_slack\Form;

use Drupal\contact_slack\Service\SlackNotificationServiceInterface;
use Drupal\contact_slack\SlackWebhookValidationTrait;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Contact Slack settings for this site.
 */
class ContactSlackSettingsForm extends ConfigFormBase {

  use SlackWebhookValidationTrait;

  /**
   * Constructs a ContactSlackSettingsForm object.
   */
  public function __construct(
    private readonly SlackNotificationServiceInterface $slackService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    // @phpstan-ignore-next-line
    return new static(
      $container->get('contact_slack.notification')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'contact_slack_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int, string>
   *   Array of editable config names.
   */
  protected function getEditableConfigNames(): array {
    return ['contact_slack.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('contact_slack.settings');

    $form['webhook'] = [
      '#type' => 'details',
      '#title' => $this->t('Webhook Configuration'),
      '#open' => TRUE,
    ];

    $form['webhook']['webhook_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default Slack Webhook URL'),
      '#description' => $this->t('Enter your Slack webhook URL. You can create one at <a href="@url" target="_blank">@url</a>.', [
        '@url' => 'https://api.slack.com/messaging/webhooks',
      ]),
      '#default_value' => $config->get('webhook_url'),
      '#required' => TRUE,
    ];

    $form['webhook']['send_test'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send Test Message'),
      '#submit' => ['::sendTestMessage'],
      '#limit_validation_errors' => [['webhook_url']],
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['retry_attempts'] = [
      '#type' => 'number',
      '#title' => $this->t('Retry Attempts'),
      '#description' => $this->t('Number of times to retry sending a failed notification.'),
      '#default_value' => $config->get('retry_attempts'),
      '#min' => 1,
      '#max' => 10,
    ];

    $form['advanced']['retry_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Retry Delay'),
      '#description' => $this->t('Delay in seconds between retry attempts.'),
      '#default_value' => $config->get('retry_delay'),
      '#min' => 1,
      '#max' => 60,
    ];

    $form['advanced']['log_failures'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log Failures'),
      '#description' => $this->t('Log failed Slack notification attempts to the system log.'),
      '#default_value' => $config->get('log_failures'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $webhook_url = $form_state->getValue('webhook_url');
    // Validate Slack webhook URL format.
    if (!empty($webhook_url) && !$this->validateSlackWebhookUrl($webhook_url)) {
      $form_state->setErrorByName('webhook_url', $this->t('Please enter a valid Slack webhook URL. It must start with https://hooks.slack.com/services/'));
    }

    // @phpstan-ignore-next-line
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('contact_slack.settings')
      ->set('webhook_url', $form_state->getValue('webhook_url'))
      ->set('retry_attempts', $form_state->getValue('retry_attempts'))
      ->set('retry_delay', $form_state->getValue('retry_delay'))
      ->set('log_failures', $form_state->getValue('log_failures'))
      ->save();

    // @phpstan-ignore-next-line
    parent::submitForm($form, $form_state);
  }

  /**
   * Submit handler for sending test messages.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function sendTestMessage(array &$form, FormStateInterface $form_state): void {
    $webhook_url = $form_state->getValue('webhook_url');

    if (empty($webhook_url)) {
      $this->messenger()->addError($this->t('Please enter a webhook URL before sending a test message.'));
      return;
    }

    // Validate Slack webhook URL format.
    if (!$this->validateSlackWebhookUrl($webhook_url)) {
      $this->messenger()->addError($this->t('Please enter a valid Slack webhook URL. It must start with https://hooks.slack.com/services/'));
      return;
    }

    // Temporarily save the webhook URL for the test.
    $this->config('contact_slack.settings')
      ->set('webhook_url', $webhook_url)
      ->save();

    if ($this->slackService->sendTestMessage()) {
      $this->messenger()->addStatus($this->t('Test message sent successfully to Slack!'));
    }
    else {
      $this->messenger()->addError($this->t('Failed to send test message. Please check your webhook URL and try again.'));
    }
  }

}
