<?php

namespace Drupal\contact_slack\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\contact\MessageInterface;
use Drupal\contact_slack\SlackWebhookValidationTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for sending Slack notifications for contact messages.
 */
class SlackNotificationService implements SlackNotificationServiceInterface {

  use SlackWebhookValidationTrait;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /**
   * Constructs a SlackNotificationService object.
   */
  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
    private readonly DateFormatterInterface $dateFormatter,
    private readonly TimeInterface $time,
    private readonly RequestStack $requestStack,
  ) {
    $this->logger = $loggerFactory->get('contact_slack');
  }

  /**
   * {@inheritdoc}
   */
  public function sendContactNotification(MessageInterface $message): bool {
    $contact_form = $message->getContactForm();

    // Get webhook URLs: use per-form URLs if set, otherwise use global default.
    $webhook_urls = $contact_form->getThirdPartySetting('contact_slack', 'webhook_urls', []);

    if (empty($webhook_urls)) {
      $config = $this->configFactory->get('contact_slack.settings');
      $default_webhook = $config->get('webhook_url');
      if (!empty($default_webhook)) {
        $webhook_urls = [$default_webhook];
      }
    }

    if (empty($webhook_urls)) {
      $this->logger->error('Slack webhook URL is not configured for form: @form', [
        '@form' => $contact_form->label(),
      ]);
      return FALSE;
    }

    // Format the message once.
    $slack_message = $this->formatMessage($message);

    // Send to all webhook URLs.
    $success = TRUE;
    foreach ($webhook_urls as $webhook_url) {
      $result = $this->sendSlackMessage($webhook_url, $slack_message);
      if (!$result) {
        $success = FALSE;
      }
    }

    return $success;
  }

  /**
   * {@inheritdoc}
   */
  public function sendTestMessage(string $channel = ''): bool {
    $config = $this->configFactory->get('contact_slack.settings');
    $webhook_url = $config->get('webhook_url');

    if (empty($webhook_url)) {
      return FALSE;
    }

    $test_message = [
      'text' => '🧪 Test message from Drupal Contact Slack module',
      'attachments' => [
        [
          'color' => 'good',
          'fields' => [
            [
              'title' => 'Status',
              'value' => 'Slack integration is working correctly!',
              'short' => FALSE,
            ],
            [
              'title' => 'Timestamp',
              'value' => $this->dateFormatter->format($this->time->getRequestTime(), 'medium'),
              'short' => TRUE,
            ],
          ],
        ],
      ],
    ];

    return $this->sendSlackMessage($webhook_url, $test_message);
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   The formatted Slack message.
   */
  public function formatMessage(MessageInterface $message): array {
    $contact_form = $message->getContactForm();
    $sender_name = $message->getSenderName();
    $sender_mail = $message->getSenderMail();
    $subject = $message->getSubject();
    $body = $message->getMessage();

    // Get form settings for which fields to include.
    $include_fields = $contact_form->getThirdPartySetting('contact_slack', 'include_fields', [
      'name', 'mail', 'subject', 'message', 'form_name',
    ]);
    $include_system_info = $contact_form->getThirdPartySetting('contact_slack', 'include_system_info', FALSE);

    // Build fields array based on settings.
    $fields = [];

    // Add sender name if enabled.
    if (!empty($include_fields['name']) && !empty($sender_name)) {
      $fields[] = [
        'title' => 'Name',
        'value' => $sender_name,
        'short' => TRUE,
      ];
    }

    // Add sender email if enabled.
    if (!empty($include_fields['mail'])) {
      $fields[] = [
        'title' => 'Email',
        'value' => $sender_mail,
        'short' => TRUE,
      ];
    }

    // Add subject if enabled.
    if (!empty($include_fields['subject'])) {
      $fields[] = [
        'title' => 'Subject',
        'value' => $subject,
        'short' => TRUE,
      ];
    }

    // Add form name if enabled.
    if (!empty($include_fields['form_name'])) {
      $fields[] = [
        'title' => 'Form',
        'value' => $contact_form->label(),
        'short' => TRUE,
      ];
    }

    // Add message body if enabled.
    if (!empty($include_fields['message'])) {
      $fields[] = [
        'title' => 'Message',
        'value' => $body,
        'short' => FALSE,
      ];
    }

    // Add system information if enabled.
    if ($include_system_info) {
      $request = $this->requestStack->getCurrentRequest();
      if ($request) {
        $fields[] = [
          'title' => 'IP Address',
          'value' => $request->getClientIp(),
          'short' => TRUE,
        ];
        $fields[] = [
          'title' => 'User Agent',
          'value' => $request->headers->get('User-Agent'),
          'short' => FALSE,
        ];
      }
    }

    // Get optional default message text.
    $default_message = $contact_form->getThirdPartySetting('contact_slack', 'default_message', '');
    
    // Build the main text with optional default message.
    $main_text = '🔔 New Contact Form Submission';
    if (!empty($default_message)) {
      $main_text = $default_message . "\n\n" . $main_text;
    }

    // Default message format.
    $slack_message = [
      'text' => $main_text,
      'attachments' => [
        [
          'color' => 'good',
          'fields' => $fields,
        ],
      ],
    ];

    return $slack_message;
  }

  /**
   * {@inheritdoc}
   */
  public function validateWebhookUrl(string $webhook_url): bool {
    return $this->validateSlackWebhookUrl($webhook_url);
  }

  /**
   * Send a message to Slack via webhook.
   *
   * @param string $webhook_url
   *   The Slack webhook URL.
   * @param array<string, mixed> $message
   *   The message data to send.
   *
   * @return bool
   *   TRUE if the message was sent successfully, FALSE otherwise.
   */
  protected function sendSlackMessage(string $webhook_url, array $message): bool {
    $config = $this->configFactory->get('contact_slack.settings');
    $retry_attempts = (int) ($config->get('retry_attempts') ?? 3);
    $retry_delay = (int) ($config->get('retry_delay') ?? 5);

    for ($attempt = 1; $attempt <= $retry_attempts; $attempt++) {
      try {
        // @phpstan-ignore-next-line
        $response = $this->httpClient->post($webhook_url, [
          'json' => $message,
          'timeout' => 10,
        ]);

        if ($response->getStatusCode() === 200) {
          $this->logger->info('Slack notification sent successfully.');
          return TRUE;
        }
      }
      catch (RequestException $e) {
        $error_message = 'Failed to send Slack notification (attempt @attempt/@max): @error';
        $this->logger->error($error_message, [
          '@attempt' => $attempt,
          '@max' => $retry_attempts,
          '@error' => $e->getMessage(),
        ]);

        // Wait before retrying (except on the last attempt).
        if ($attempt < $retry_attempts) {
          sleep($retry_delay);
        }
      }
    }

    return FALSE;
  }

}
