<?php

namespace Drupal\Tests\contact_slack\Kernel;

use Drupal\contact\Entity\ContactForm;
use Drupal\contact\Entity\Message;
use Drupal\KernelTests\KernelTestBase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;

/**
 * Integration test for complete Slack notification workflow.
 *
 * This test verifies the end-to-end integration of contact messages
 * with Slack notifications, including:
 * - Message creation and formatting
 * - Per-form configuration
 * - Multiple webhook URL handling
 * - Field selection
 * - System info inclusion
 * - Error handling
 *
 * Unlike unit tests which mock everything, this test runs in a real
 * Drupal environment to ensure all components work together correctly.
 *
 * @group contact_slack
 */
class SlackNotificationIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'contact',
    'contact_slack',
    'user',
    'system',
    'field',
  ];

  /**
   * The Slack notification service.
   *
   * @var \Drupal\contact_slack\Service\SlackNotificationServiceInterface
   */
  protected $slackService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('contact_message');
    $this->installEntitySchema('user');
    $this->installConfig(['contact', 'contact_slack', 'system']);

    $this->slackService = $this->container->get('contact_slack.notification');
  }

  /**
   * Tests complete notification workflow with per-form webhook URLs.
   */
  public function testCompleteNotificationWorkflow(): void {
    // Create contact form with Slack enabled.
    $contactForm = ContactForm::create([
      'id' => 'feedback',
      'label' => 'Feedback Form',
      'recipients' => ['admin@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', TRUE);
    $contactForm->setThirdPartySetting('contact_slack', 'webhook_urls', [
      'https://hooks.slack.com/services/TEST123/ABC456/xyz789',
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'include_fields', [
      'name', 'mail', 'subject', 'message',
    ]);
    $contactForm->save();

    // Mock HTTP client to capture the Slack API call.
    $mock = new MockHandler([
      new Response(200, [], 'ok'),
    ]);
    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);
    $this->container->set('http_client', $client);

    // Recreate service with mocked client.
    $this->slackService = $this->container->get('contact_slack.notification');

    // Create and send contact message.
    $message = Message::create([
      'contact_form' => 'feedback',
      'name' => 'John Doe',
      'mail' => 'john@example.com',
      'subject' => 'Test Feedback',
      'message' => 'This is a test message.',
    ]);

    $result = $this->slackService->sendContactNotification($message);

    $this->assertTrue($result, 'Notification should be sent successfully');
  }

  /**
   * Tests notification with multiple webhook URLs.
   */
  public function testMultipleWebhookUrls(): void {
    $contactForm = ContactForm::create([
      'id' => 'support',
      'label' => 'Support Form',
      'recipients' => ['support@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', TRUE);
    $contactForm->setThirdPartySetting('contact_slack', 'webhook_urls', [
      'https://hooks.slack.com/services/URL1/URL2/token1',
      'https://hooks.slack.com/services/URL3/URL4/token2',
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'include_fields', ['name', 'mail']);
    $contactForm->save();

    // Mock two successful responses.
    $mock = new MockHandler([
      new Response(200, [], 'ok'),
      new Response(200, [], 'ok'),
    ]);
    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);
    $this->container->set('http_client', $client);

    $this->slackService = $this->container->get('contact_slack.notification');

    $message = Message::create([
      'contact_form' => 'support',
      'name' => 'Jane Smith',
      'mail' => 'jane@example.com',
      'subject' => 'Support Request',
      'message' => 'Need help.',
    ]);

    $result = $this->slackService->sendContactNotification($message);

    $this->assertTrue($result, 'Notification to multiple webhooks should succeed');
  }

  /**
   * Tests notification falls back to default webhook URL.
   */
  public function testFallbackToDefaultWebhook(): void {
    // Set default webhook URL.
    $this->config('contact_slack.settings')
      ->set('webhook_url', 'https://hooks.slack.com/services/DEFAULT/URL/token')
      ->save();

    // Create form without custom webhook.
    $contactForm = ContactForm::create([
      'id' => 'general',
      'label' => 'General Form',
      'recipients' => ['info@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', TRUE);
    $contactForm->setThirdPartySetting('contact_slack', 'webhook_urls', []);
    $contactForm->save();

    $mock = new MockHandler([
      new Response(200, [], 'ok'),
    ]);
    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);
    $this->container->set('http_client', $client);

    $this->slackService = $this->container->get('contact_slack.notification');

    $message = Message::create([
      'contact_form' => 'general',
      'name' => 'Test User',
      'mail' => 'test@example.com',
      'subject' => 'Test',
      'message' => 'Testing default webhook.',
    ]);

    $result = $this->slackService->sendContactNotification($message);

    $this->assertTrue($result, 'Should use default webhook URL');
  }

  /**
   * Tests disabled form does not send notification.
   */
  public function testDisabledFormDoesNotSend(): void {
    $contactForm = ContactForm::create([
      'id' => 'disabled',
      'label' => 'Disabled Form',
      'recipients' => ['admin@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', FALSE);
    $contactForm->save();

    $message = Message::create([
      'contact_form' => 'disabled',
      'name' => 'Test User',
      'mail' => 'test@example.com',
      'subject' => 'Should not send',
      'message' => 'This should not trigger notification.',
    ]);

    // Verify form is disabled.
    $enabled = $message->getContactForm()->getThirdPartySetting('contact_slack', 'enabled', FALSE);
    $this->assertFalse($enabled, 'Form should be disabled');
  }

  /**
   * Tests message formatting with field selection.
   */
  public function testMessageFormattingWithFieldSelection(): void {
    $contactForm = ContactForm::create([
      'id' => 'selective',
      'label' => 'Selective Form',
      'recipients' => ['admin@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', TRUE);
    $contactForm->setThirdPartySetting('contact_slack', 'webhook_urls', [
      'https://hooks.slack.com/services/TEST/TEST/test',
    ]);
    // Only include name and subject.
    $contactForm->setThirdPartySetting('contact_slack', 'include_fields', [
      'name', 'subject',
    ]);
    $contactForm->save();

    $message = Message::create([
      'contact_form' => 'selective',
      'name' => 'John Doe',
      'mail' => 'john@example.com',
      'subject' => 'Test Subject',
      'message' => 'This message should not appear in Slack.',
    ]);

    $formatted = $this->slackService->formatMessage($message);

    $fields = $formatted['attachments'][0]['fields'];
    $fieldTitles = array_column($fields, 'title');

    // Should include name and subject.
    $this->assertContains('Name', $fieldTitles, 'Name field should be included');
    $this->assertContains('Subject', $fieldTitles, 'Subject field should be included');

    // Should NOT include email or message.
    $this->assertNotContains('Email', $fieldTitles, 'Email field should not be included');
    $this->assertNotContains('Message', $fieldTitles, 'Message field should not be included');
  }

  /**
   * Tests error handling when webhook fails.
   */
  public function testWebhookFailureHandling(): void {
    $contactForm = ContactForm::create([
      'id' => 'error_test',
      'label' => 'Error Test Form',
      'recipients' => ['admin@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', TRUE);
    $contactForm->setThirdPartySetting('contact_slack', 'webhook_urls', [
      'https://hooks.slack.com/services/FAIL/TEST/error',
    ]);
    $contactForm->save();

    // Mock failed request.
    $mock = new MockHandler([
      new RequestException(
        'Error Communicating with Server',
        new Request('POST', 'test'),
        new Response(500, [], 'Server Error')
      ),
    ]);
    $handlerStack = HandlerStack::create($mock);
    $client = new Client(['handler' => $handlerStack]);
    $this->container->set('http_client', $client);

    $this->slackService = $this->container->get('contact_slack.notification');

    $message = Message::create([
      'contact_form' => 'error_test',
      'name' => 'Test User',
      'mail' => 'test@example.com',
      'subject' => 'Error Test',
      'message' => 'Testing error handling.',
    ]);

    $result = $this->slackService->sendContactNotification($message);

    // Should return FALSE on failure.
    $this->assertFalse($result, 'Should return FALSE when webhook fails');
  }

  /**
   * Tests system info inclusion in message.
   */
  public function testSystemInfoInclusion(): void {
    $contactForm = ContactForm::create([
      'id' => 'sysinfo',
      'label' => 'System Info Form',
      'recipients' => ['admin@example.com'],
    ]);
    $contactForm->setThirdPartySetting('contact_slack', 'enabled', TRUE);
    $contactForm->setThirdPartySetting('contact_slack', 'include_fields', ['name']);
    $contactForm->setThirdPartySetting('contact_slack', 'include_system_info', TRUE);
    $contactForm->save();

    $message = Message::create([
      'contact_form' => 'sysinfo',
      'name' => 'Test User',
      'mail' => 'test@example.com',
      'subject' => 'Test',
      'message' => 'Testing system info.',
    ]);

    $formatted = $this->slackService->formatMessage($message);
    $fields = $formatted['attachments'][0]['fields'];
    $fieldTitles = array_column($fields, 'title');

    // Should include system info fields.
    $this->assertContains('IP Address', $fieldTitles, 'IP Address should be included');
    $this->assertContains('User Agent', $fieldTitles, 'User Agent should be included');
  }

}
