<?php

namespace Drupal\Tests\contact_slack\Unit;

use Drupal\contact_slack\SlackWebhookValidationTrait;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for SlackWebhookValidationTrait.
 *
 * @group contact_slack
 * @coversDefaultClass \Drupal\contact_slack\SlackWebhookValidationTrait
 */
class SlackWebhookValidationTraitTest extends UnitTestCase {

  /**
   * The test object using the trait.
   *
   * @var object
   */
  protected $traitObject;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->traitObject = new class {
      use SlackWebhookValidationTrait {
        validateSlackWebhookUrl as public;
      }

    };
  }

  /**
   * Tests validation with valid webhook URLs.
   *
   * @covers ::validateSlackWebhookUrl
   * @dataProvider validWebhookUrlProvider
   */
  public function testValidateSlackWebhookUrlValid(string $url): void {
    $result = $this->traitObject->validateSlackWebhookUrl($url);
    $this->assertTrue($result, "URL should be valid: $url");
  }

  /**
   * Tests validation with invalid webhook URLs.
   *
   * @covers ::validateSlackWebhookUrl
   * @dataProvider invalidWebhookUrlProvider
   */
  public function testValidateSlackWebhookUrlInvalid(string $url): void {
    $result = $this->traitObject->validateSlackWebhookUrl($url);
    $this->assertFalse($result, "URL should be invalid: $url");
  }

  /**
   * Data provider for valid webhook URLs.
   *
   * @return array<string, array<string>>
   *   Valid webhook URLs.
   */
  public static function validWebhookUrlProvider(): array {
    return [
      'standard format' => ['https://hooks.slack.com/services/T00000000/B00000000/XXXXXXXXXXXXXXXXXXXX'],
      'uppercase letters' => ['https://hooks.slack.com/services/ABC123DEF/GHI456JKL/MNO789PQR'],
      'mixed case token' => ['https://hooks.slack.com/services/ABCDEF/GHIJKL/abcdefghijklmnop'],
      'long token' => ['https://hooks.slack.com/services/T1234567890/B1234567890/abcdefghijklmnopqrstuvwxyz'],
      'numbers only' => ['https://hooks.slack.com/services/123456789/987654321/123456789'],
      'all caps first two' => ['https://hooks.slack.com/services/ABCDEFGHIJ/KLMNOPQRST/uvwxyz123'],
    ];
  }

  /**
   * Data provider for invalid webhook URLs.
   *
   * @return array<string, array<string>>
   *   Invalid webhook URLs.
   */
  public static function invalidWebhookUrlProvider(): array {
    return [
      'http instead of https' => ['http://hooks.slack.com/services/ABC123/DEF456/xyz789'],
      'wrong domain' => ['https://example.com/services/ABC123/DEF456/xyz789'],
      'wrong subdomain' => ['https://api.slack.com/services/ABC123/DEF456/xyz789'],
      'missing services path' => ['https://hooks.slack.com/ABC123/DEF456/xyz789'],
      'wrong path' => ['https://hooks.slack.com/webhooks/ABC123/DEF456/xyz789'],
      'too few segments' => ['https://hooks.slack.com/services/ABC123/DEF456'],
      'too many segments' => ['https://hooks.slack.com/services/ABC123/DEF456/xyz789/extra'],
      'lowercase in first segment' => ['https://hooks.slack.com/services/abc123/DEF456/xyz789'],
      'lowercase in second segment' => ['https://hooks.slack.com/services/ABC123/def456/xyz789'],
      'special chars in path' => ['https://hooks.slack.com/services/ABC-123/DEF456/xyz789'],
      'special chars in token' => ['https://hooks.slack.com/services/ABC123/DEF456/xyz-789'],
      'slash at end' => ['https://hooks.slack.com/services/ABC123/DEF456/xyz789/'],
      'query parameters' => ['https://hooks.slack.com/services/ABC123/DEF456/xyz789?foo=bar'],
      'fragment' => ['https://hooks.slack.com/services/ABC123/DEF456/xyz789#anchor'],
      'empty string' => [''],
      'just domain' => ['https://hooks.slack.com'],
      'just https' => ['https://'],
      'no protocol' => ['hooks.slack.com/services/ABC123/DEF456/xyz789'],
      'spaces in url' => ['https://hooks.slack.com/services/ABC 123/DEF456/xyz789'],
    ];
  }

}
