<?php

namespace Drupal\contact_storage_export\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Drupal\Core\Url;
use Drupal\file\Entity\File;

/**
 * The CSV Export Download Form.
 *
 * @package Drupal\contact_storage_export\Form
 */
class ContactStorageDownloadForm extends FormBase {

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The tempstore.
   *
   * @var \Drupal\Core\TempStore\SharedTempStore
   */
  protected $tempStore;

  /**
   * Constructs a ContactStorageDownloadForm object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory.
   */
  public function __construct(MessengerInterface $messenger, PrivateTempStoreFactory $temp_store_factory) {
    $this->messenger = $messenger;
    $this->tempStore = $temp_store_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
      $container->get('tempstore.private')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'contact_storage_download_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $contact_form = '', $key = 0) {

    $form['contact_form'] = [
      '#type' => 'hidden',
      '#value' => $contact_form,
    ];
    $form['key'] = [
      '#type' => 'hidden',
      '#value' => $key,
    ];

    if ($contact_form) {

      $form['intro'] = [
        '#type' => 'item',
        '#plain_text' => $this->t('Your export is ready for download.'),
      ];

      $form['download_container'] = [
        '#type' => 'container',
      ];
      $form['download_container']['download'] = [
        '#type' => 'submit',
        '#value' => $this->t('Download'),
        '#attributes' => [
          'class' => [
            'button',
            'button--primary',
          ],
        ],
      ];

      $form['return'] = [
        '#title' => $this->t('Return to the export page.'),
        '#type' => 'link',
        '#url' => Url::fromRoute('entity.contact_form.export_form'),
      ];

    }
    else {

      $message = $this->t('An unknown error occurred preparing your download.');
      $this->messenger->addWarning($message, FALSE);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $values = $form_state->getValues();

    // Get data from tempstore.
    $tempstore = $this->tempStore->get('contact_storage_export');
    $data = $tempstore->get('data');

    if (isset($data[$values['key']])) {
      $export = $data[$values['key']];

      // Load the file.
      if ($file = File::load($export['fid'])) {
        // Send the export file.
        $response = new BinaryFileResponse($file->getFileUri());
        $response->setContentDisposition('attachment', $export['filename']);
        $form_state->setResponse($response);
      }
      else {
        $message = $this->t('Failed to load the file.');
        $this->messenger->addWarning($message);
      }

    }
    else {
      $message = $this->t('Failed to download the file.');
      $this->messenger->addWarning($message);
    }

  }

}
