<?php

namespace Drupal\Tests\contact_storage_export\Kernel;

use Drupal\contact\Entity\ContactForm;
use Drupal\contact\Entity\Message;
use Drupal\contact_storage_export\ContactStorageExportBatches;
use Drupal\file\Entity\File;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests contact storage export service methods.
 *
 * @group contact_storage
 */
class ContactStorageExportBatchesKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'field',
    'contact_storage',
    'contact_storage_export',
    'csv_serialization',
    'contact',
    'user',
    'system',
    'file',
  ];

  /**
   * The service under test.
   *
   * @var \Drupal\contact_storage_export\ContactStorageExportBatches
   */
  protected $exportBatches;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('contact_message');
    $this->installEntitySchema('file');
    $this->installConfig(['field', 'system']);

    // Initialize service.
    $this->exportBatches = new ContactStorageExportBatches();
  }

  /**
   * Tests contact storage export batch.
   *
   * @requires module contact_storage
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testBatchOperations() {
    // Create a sample form.
    $num_messages = 30;
    $contact_form_id = 'contact_storage_export_form';
    $contact_form = ContactForm::create(['id' => $contact_form_id]);
    $contact_form->save();

    // Create a sample messages.
    for ($i = 1; $i <= $num_messages; $i++) {
      $message = Message::create([
        'id' => $i,
        'contact_form' => $contact_form->id(),
        'name' => 'example',
        'mail' => 'admin@example.com',
        'ip_address' => '127.0.0.1',
      ]);
      $message->save();
    }

    // Simulate batch steps.
    $settings = [
      'contact_form' => $contact_form_id,
      'since_last_export' => '',
    ];
    $context = [];
    do {
      $this->exportBatches::processBatch($settings, $context);
    } while ($context['sandbox']['progress'] < $context['sandbox']['max']);

    // Load temp file.
    $tempFile = File::load($context['results']['fid']);
    $lines = file($tempFile->getFileUri(), FILE_IGNORE_NEW_LINES);

    // Expect 1 line with header and number of  messages inserted.
    $this->assertCount(1 + $num_messages, $lines);

    // Expect header to be first line and does not repeat.
    $header = '"Message ID",Language,"Form ID","The sender\'s name","The sender\'s email",Subject,Message,Copy,"Recipient ID",Created,"User ID","IP address"';
    $this->assertEquals($header, $lines[0], "The header should be the first line");
    $count = array_count_values($lines);
    $this->assertEquals(1, $count[$header] ?? 0, "The header should appear exactly once.");
  }

}
