<?php

namespace Drupal\contacts_group\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form for an organisation relationship.
 */
class ContactOrgRelationshipForm extends ContentEntityForm {

  /**
   * The Group Content entity.
   *
   * @var \Drupal\group\Entity\GroupRelationshipInterface
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $group = $this->entity->getGroup();
    $member = $this->entity->getEntity();

    // If this is a join form with predefined entities, check that the
    // relationship doesn't already exist.
    if ($group && $member) {
      $plugin = $this->entity->getPlugin();
      $existing_relationships = $group->getRelationshipsByEntity($member, $plugin->getPluginId());
      if ($this->entity->isNew() && !empty($existing_relationships)) {
        if ($member->id() == $this->currentUser()->id()) {
          $member_label = $this->t('You');
        }
        else {
          $member_label = new FormattableMarkup('%label', ['%label' => $member->label()]);
        }
        return [
          '#markup' => $this->t('@member have already joined %group.', [
            '@member' => $member_label,
            '%group' => $group->label(),
          ]),
        ];
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $group = $this->entity->getGroup();
    $content = $this->entity->getEntity();

    // Show a selection for the organisation for a new relationship.
    if (!$group) {
      $form['organisation'] = [
        '#type' => 'entity_autocomplete',
        '#required' => TRUE,
        '#title' => $this->t('Organisation'),
        '#target_type' => 'user',
        '#selection_handler' => 'search_api_user_multiline',
        '#selection_settings' => [
          'index' => 'contacts_index',
        ],
        '#weight' => -99,
      ];

      // If the form state has been set up to restrict to certain roles from the
      // organisation finder (e.g. a subset of organisations) apply the
      // conditions to the selection.
      if ($organisation_roles = $form_state->get('organisation_roles')) {
        $form['organisation']['#selection_settings']['conditions'][] = [
          'roles',
          $organisation_roles,
          'IN',
        ];
      }

      // Prevent joining yourself.
      if ($content) {
        $form['organisation']['#selection_settings']['conditions'][] = [
          'uid',
          $content->id(),
          '<>',
        ];

        // Don't allow contacts to be added to organisations they are already a
        // member of.
        if ($content->hasField('organisations')) {
          foreach ($content->get('organisations')->getValue() as $value) {
            $form['organisation']['#selection_settings']['conditions'][] = [
              'uid',
              $value['target_id'],
              '<>',
            ];
          }
        }
      }
    }

    // Show selection for the member for a new relationship.
    if (!$content) {
      $element = &$form['entity_id']['widget'][0]['target_id'];

      // Adjust the existing form element.
      $element['#title'] = $this->t('Member');
      unset($element['#description']);
      $element['#selection_handler'] = 'search_api_user_multiline';
      $element['#selection_settings'] = [
        'index' => 'contacts_index',
      ];

      // If the form state has been set up to restrict to certain roles from the
      // member finder (e.g. only individuals or only organisations) apply the
      // conditions to the selection.
      if ($roles = $form_state->get('content_roles')) {
        $element['#selection_settings']['conditions'][] = [
          'roles',
          $roles,
          'IN',
        ];

        if (in_array('crm_indiv', $roles) && !in_array('crm_org', $roles)) {
          if (isset($form['indiv_role']) && !$form['indiv_role']['#access']) {
            // If access to indiv_role has been denied by
            // contacts_group_entity_field_access then we want to re-enable it
            // as it's safe to show it on this screen as we're not working with
            // org-to-org relationships.
            $form['indiv_role']['#access'] = TRUE;
          }
        }
      }

      // Prevent adding yourself.
      if ($group) {
        $element['#selection_settings']['conditions'][] = [
          'uid',
          $group->contacts_org->target_id,
          '<>',
        ];
      }
    }
    // Don't allow changing the member of existing relationships.
    else {
      // Hide the user selection.
      $form['entity_id']['#access'] = FALSE;
    }

    // Hide the roles field if there are no roles.
    if (empty($form['group_roles']['widget']['#options'])) {
      $form['group_roles']['#access'] = FALSE;
    }

    // If we have the bypass group access permission, override the access on the
    // group role field.
    // @todo Replace this by patching group_entity_field_access to check the
    // bypass permission before it forbids.
    elseif ($this->currentUser()->hasPermission('bypass group access')) {
      $form['group_roles']['#access'] = TRUE;
    }

    // If there are no indiv role options, hide the field.
    if (isset($form['indiv_role'])) {
      if (empty($form['indiv_role']['widget']['#options']) || (count($form['indiv_role']['widget']['#options']) == 1 && isset($form['indiv_role']['widget']['#options']['_none']))) {
        $form['indiv_role']['#access'] = FALSE;
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    unset($actions['delete']);
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function buildEntity(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\group\Entity\GroupRelationshipInterface $entity */
    $entity = parent::buildEntity($form, $form_state);

    if ($entity->isNew()) {
      $org_id = $form_state->getValue('organisation');
      if ($org_id) {
        $org = $this->entityTypeManager->getStorage('user')->load($org_id);
        $form_state->set('group', $org->group->entity);
      }
    }

    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    if ($group = $form_state->get('group')) {
      $group->save();
      $this->entity->set('gid', $group->id());
    }
    return parent::save($form, $form_state);
  }

}
