<?php

namespace Drupal\contacts\Plugin\Validation\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates that an entity reference field is unique.
 */
class UniqueReferenceConstraintValidator extends ConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($items, Constraint $constraint) {
    if (!$item = $items->first()) {
      return;
    }
    $field_name = $items->getFieldDefinition()->getName();
    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    $entity = $items->getEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $entity_type = $entity->getEntityType();
    $id_key = $entity_type->getKey('id');

    $query = \Drupal::entityQuery($entity_type_id)
      // The id could be NULL, so we cast it to 0 in that case.
      ->accessCheck()
      ->condition($id_key, (int) $items->getEntity()->id(), '<>')
      ->condition($field_name, $item->target_id)
      ->range(0, 1);

    if ($constraint->bundle && ($bundle_key = $entity_type->getKey('bundle'))) {
      $query->condition($bundle_key, $entity->bundle());
    }

    $value_taken = (bool) $query
      ->accessCheck()
      ->count()
      ->execute();

    if ($value_taken) {
      $this->context->addViolation($constraint->message, [
        '%value' => $item->entity->label(),
        '@entity_type' => $entity->getEntityType()->getSingularLabel(),
        '@field_name' => mb_strtolower($items->getFieldDefinition()->getLabel()),
      ]);
    }
  }

}
