<?php

namespace Drupal\content_archiver\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

class ArchiverController extends ControllerBase {
	protected $messenger;

	public function __construct(MessengerInterface $messenger) {
		$this->messenger = $messenger;
	}

	public static function create(ContainerInterface $container) {
		return new static(
			$container->get('messenger')
		);
	}

	public function run(): RedirectResponse {
		$config = $this->config('content_archiver.settings');
		$field_name = $config->get('expiration_field') ?: 'field_expiration';
		$notify = (bool) $config->get('notify_author');
		$subject = $config->get('notification_subject') ?: 'Your content has been archived';
		$body = $config->get('notification_body') ?: 'Hello [author:name],\n\nYour content "[node:title]" has been archived as of [date].';
		$batch_size = (int) ($config->get('batch_size') ?? 50);

		$now = new DrupalDateTime('now');
		$now_ts = $now->getTimestamp();
		$formatted_date = \Drupal::service('date.formatter')->format($now_ts, 'custom', 'Y-m-d H:i');

		$storage = $this->entityTypeManager()->getStorage('node');
		$query = $storage->getQuery()
			->condition('status', 1)
			->exists($field_name)
			->condition($field_name, $now->format('Y-m-d\TH:i:s'), '<=')
			->range(0, $batch_size)
			->accessCheck(FALSE);
		$nids = $query->execute();

		$processed = 0;
		if (!empty($nids)) {
			$language_id = $this->languageManager()->getDefaultLanguage()->getId();
			$mail_manager = \Drupal::service('plugin.manager.mail');
			foreach ($nids as $nid) {
				/** @var \Drupal\node\NodeInterface|null $node */
				$node = Node::load($nid);
				if ($node && $node->isPublished()) {
					$node->setUnpublished();
					$node->save();
					$processed++;
					if ($notify) {
						$author = $node->getOwner();
						if ($author && $author->isActive() && $author->getEmail()) {
							$params = [
								'subject' => $subject,
								'body' => $this->buildBody($body, $author->getDisplayName(), $node->label(), $formatted_date),
							];
							$mail_manager->mail('content_archiver', 'content_archived', $author->getEmail(), $language_id, $params);
						}
					}
				}
			}
		}

		$this->messenger->addStatus($this->t('Content Archiver processed @count item(s).', ['@count' => $processed]));
		return $this->redirect('entity.node_type.edit_form', ['node_type' => 'blog']);
	}

	protected function buildBody(string $template, string $author_name, string $node_title, string $date): string {
		$replacements = [
			'[author:name]' => $author_name,
			'[node:title]' => $node_title,
			'[date]' => $date,
		];
		return strtr($template ?: 'Hello [author:name],\n\nYour content "[node:title]" has been archived as of [date].', $replacements);
	}
} 